<?php
// ============================================================================
// TLibrary — Forbidden Magicka Archive of The Republic 📚
// Location: /tlibrary.php
//
// GOLD GOD SEED: one file, full library.
//
// On Sync Bell (any citizen):
//   • Loads local Atom "GOD file":
//       - /feed.atom   (Blogger Takeout export, stored on *your* server)
//   • For each <entry> where blogger:type=POST & blogger:status=LIVE:
//       - Reads blogger:filename for canonical slug:
//
//           /2025/01/emergent-learning-new-approach-to.html
//           → emergent-learning-new-approach-to
//
//       - Reads:
//           • <title>
//           • <content type="html"> (HTML ENTITIES DECODED)
//           • <published>, <updated>, blogger:created
//           • <category term="..."> → tags[]
//           • blogger:metaDescription (when present)
//       - Decides kind: book / law / chronicle (from tags)
//       - Writes /tlibrary/SLUG.php TShell page with full HTML body
//       - Adds entry to tlibrary-index.json (for the JS console)
//
// All flat files. No SQL. No external HTTP. Just Seeds.
//
// Authored for The Republic by:
//   Codey, Republic Systems Programmer 💻👑
// ============================================================================

// ---------------------------------------------------------------------------
// 0. CONFIG
// ---------------------------------------------------------------------------

$TLIBRARY_BASE_URL    = 'https://trepublic.net';
$TLIBRARY_CONTENT_DIR = __DIR__ . '/tlibrary';
$TLIBRARY_INDEX_JSON  = __DIR__ . '/tlibrary/admin/data/tlibrary-index.json';
// LOCAL GOD FILE ONLY: exported feed.atom lives beside the tomes
$TLIBRARY_LOCAL_FEED  = __DIR__ . '/feed.atom';

// ---------------------------------------------------------------------------
// 1. HELPER FUNCTIONS
// ---------------------------------------------------------------------------

/**
 * Normalise string to a slug: lowercase, dash-separated, safe characters.
 */
function tlibrary_slugify(string $str): string
{
    $str = strtolower($str);
    $str = preg_replace('/[^a-z0-9]+/u', '-', $str);
    $str = trim($str, '-');
    if ($str === '') {
        $str = 'tome';
    }
    return $str;
}

/**
 * Build slug from blogger:filename, or fallback to link/title/id.
 *
 * WE KEEP BLOGGER SLUGS (no date prefix).
 *
 * Examples:
 *   /2025/01/emergent-learning-new-approach-to.html
 *     → emergent-learning-new-approach-to
 *   /2020/11/open-source-university.html
 *     → open-source-university
 */
function tlibrary_make_slug(
    string $bloggerFilename,
    string $href,
    string $title,
    string $idUri
): string {
    // 1) blogger:filename like /2025/01/emergent-learning-new-approach-to.html
    if ($bloggerFilename !== '') {
        $path = trim($bloggerFilename);
        $path = ltrim($path, '/');

        if (preg_match('~^\d{4}/\d{2}/([^/]+)\.html$~', $path, $m)) {
            $base = $m[1];
            return tlibrary_slugify($base);
        }

        $base = basename($path, '.html');
        if ($base !== '' && $base !== '/') {
            return tlibrary_slugify($base);
        }
    }

    // 2) Fallback: permalink href path
    if ($href !== '') {
        $parts = parse_url($href);
        if (!empty($parts['path'])) {
            $path = ltrim($parts['path'], '/');

            if (preg_match('~^\d{4}/\d{2}/([^/]+)\.html$~', $path, $m)) {
                $base = $m[1];
                return tlibrary_slugify($base);
            }

            $base = basename($path, '.html');
            if ($base !== '' && $base !== '/') {
                return tlibrary_slugify($base);
            }
        }
    }

    // 3) Title
    if ($title !== '') {
        return tlibrary_slugify($title);
    }

    // 4) <id> URI
    if ($idUri !== '') {
        $base = basename($idUri);
        return tlibrary_slugify($base);
    }

    // Last resort
    return 'tome-' . uniqid();
}

/**
 * Export meta array as valid PHP code.
 */
function tlibrary_export_meta_php(array $meta): string
{
    return var_export($meta, true);
}

/**
 * Escape a PHP single-quoted string.
 */
function tlibrary_php_string(string $value): string
{
    $value = str_replace(['\\', '\''], ['\\\\', '\\\''], $value);
    return "'" . $value . "'";
}

/**
 * Load Atom GOD file from local disk only.
 */
function tlibrary_load_feed_xml(?string &$outError = null): ?SimpleXMLElement
{
    global $TLIBRARY_LOCAL_FEED;

    if (!is_file($TLIBRARY_LOCAL_FEED)) {
        $outError = '❌ Local GOD file not found at ' . $TLIBRARY_LOCAL_FEED .
                    ' — upload your Blogger Takeout feed.atom there.';
        return null;
    }

    $xmlString = @file_get_contents($TLIBRARY_LOCAL_FEED);
    if ($xmlString === false) {
        $outError = '❌ Could not read local GOD file ' . $TLIBRARY_LOCAL_FEED . '.';
        return null;
    }

    $xmlString = trim($xmlString);
    if ($xmlString === '') {
        $outError = '❌ Local GOD file is empty.';
        return null;
    }

    libxml_use_internal_errors(true);
    $xml = simplexml_load_string($xmlString);
    if ($xml === false) {
        $outError = '❌ Local GOD file is not valid XML.';
        return null;
    }

    return $xml;
}

/**
 * Decide kind (book / law / chronicle) from tags.
 */
function tlibrary_kind_from_tags(array $tags): string
{
    $lower = array_map('strtolower', $tags);

    foreach ($lower as $t) {
        if ($t === 'book' || strpos($t, ' book') !== false) {
            return 'book';
        }
    }
    foreach ($lower as $t) {
        if ($t === 'law' || strpos($t, ' law') !== false || strpos($t, 'tlicense') !== false) {
            return 'law';
        }
    }
    return 'chronicle';
}

// ---------------------------------------------------------------------------
// 2. SYNC ENGINE
// ---------------------------------------------------------------------------

/**
 * Sync from Atom GOD file:
 *   • Ensures directories
 *   • Generates /tlibrary/SLUG.php if missing
 *   • Writes tlibrary-index.json (newest first)
 */
function tlibrary_sync_from_feed(?string &$outMessage = null): bool
{
    global $TLIBRARY_CONTENT_DIR, $TLIBRARY_INDEX_JSON, $TLIBRARY_BASE_URL;

    // Ensure directories exist
    if (!is_dir($TLIBRARY_CONTENT_DIR)) {
        @mkdir($TLIBRARY_CONTENT_DIR, 0755, true);
    }
    $indexDir = dirname($TLIBRARY_INDEX_JSON);
    if (!is_dir($indexDir)) {
        @mkdir($indexDir, 0755, true);
    }

    $xmlError = null;
    $xml      = tlibrary_load_feed_xml($xmlError);
    if (!$xml) {
        $outMessage = $xmlError;
        return false;
    }

    if (!isset($xml->entry)) {
        $outMessage = '❌ GOD file has no <entry> elements.';
        return false;
    }

    $entries       = $xml->entry;
    $index         = [];
    $createdFiles  = 0;
    $totalEntries  = 0;

    foreach ($entries as $entry) {
        $totalEntries++;

        // Blogger namespace (still used inside the local file)
        $b = $entry->children('http://schemas.google.com/blogger/2018');

        $bloggerType    = isset($b->type)    ? trim((string)$b->type)    : '';
        $bloggerStatus  = isset($b->status)  ? trim((string)$b->status)  : '';
        $bloggerCreated = isset($b->created) ? trim((string)$b->created) : '';
        $bloggerMeta    = isset($b->metaDescription) ? trim((string)$b->metaDescription) : '';
        $bloggerLoc     = isset($b->location) ? trim((string)$b->location) : '';
        $bloggerFile    = isset($b->filename) ? trim((string)$b->filename) : '';
        $bloggerTrashed = isset($b->trashed) ? trim((string)$b->trashed) : '';

        // Only POST + LIVE (ignore drafts, pages, deleted)
        if ($bloggerType !== '' && strtoupper($bloggerType) !== 'POST') {
            continue;
        }
        if ($bloggerStatus !== '' && strtoupper($bloggerStatus) !== 'LIVE') {
            continue;
        }

        $title        = trim((string)$entry->title);
        $idUri        = (string)$entry->id;
        $publishedStr = (string)$entry->published;
        $updatedStr   = (string)$entry->updated;

        $publishedTs = 0;
        if ($publishedStr !== '') {
            $publishedTs = strtotime($publishedStr);
        }
        if (!$publishedTs && $updatedStr !== '') {
            $publishedTs = strtotime($updatedStr);
        }
        $publishedTs   = $publishedTs ?: 0;
        $publishedIso  = $publishedTs > 0 ? gmdate('Y-m-d\TH:i:s\Z', $publishedTs) : '';
        $publishedDate = $publishedTs > 0 ? gmdate('Y-m-d', $publishedTs) : '';

        $createdIso = '';
        if ($bloggerCreated !== '') {
            $createdTs = strtotime($bloggerCreated);
            if ($createdTs) {
                $createdIso = gmdate('Y-m-d\TH:i:s\Z', $createdTs);
            }
        }

        // Pick alternate link (for potential slug fallback)
        $altHref = '';
        foreach ($entry->link as $link) {
            $attrs = $link->attributes();
            $rel   = (string)$attrs['rel'];
            $href  = (string)$attrs['href'];
            if ($rel === '' || $rel === 'alternate') {
                if ($href !== '') {
                    $altHref = $href;
                    break;
                }
            }
        }

        // Slug via blogger:filename first, Blogger-style
        $slug = tlibrary_make_slug($bloggerFile, $altHref, $title, $idUri);

        // Tags from <category term="...">
        $tags = [];
        foreach ($entry->category as $cat) {
            $term = trim((string)$cat['term']);
            if ($term !== '') {
                $tags[$term] = true;
            }
        }
        $tagsList = array_keys($tags); // avoid "true" keys

        $kind  = tlibrary_kind_from_tags($tagsList);
        $shelf = '';
        foreach ($tagsList as $t) {
            if (stripos($t, 'foundational 11') !== false) {
                $shelf = 'Foundational 11';
                break;
            }
        }

        // Content HTML: decode entities
        $contentHtml = '';
        if (isset($entry->content)) {
            $raw = (string)$entry->content;
            $contentHtml = html_entity_decode($raw, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        }
        if ($contentHtml === '' && isset($entry->summary)) {
            $raw = (string)$entry->summary;
            $contentHtml = html_entity_decode($raw, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        }

        // Generate tome PHP file if missing
        $filePath = $TLIBRARY_CONTENT_DIR . '/' . $slug . '.php';
        if (!is_file($filePath)) {
            $pageTitle = $title !== '' ? $title : 'TLibrary Tome';

            // Snippet from plain text
            $plain   = trim(strip_tags($contentHtml));
            $plain   = preg_replace('/\s+/u', ' ', $plain);
            $snippet = mb_substr($plain, 0, 220, 'UTF-8');
            if (mb_strlen($plain, 'UTF-8') > 220) {
                $snippet .= '…';
            }
            if ($snippet === '') {
                $snippet = 'A tome from the Bard President archive of The Republic.';
            }
            // Override with blogger:metaDescription when present
            if ($bloggerMeta !== '') {
                $snippet = $bloggerMeta;
            }

            $meta = [
                'id'              => $slug,
                'slug'            => $slug,
                'kind'            => $kind,
                'title'           => $pageTitle,
                'published'       => $publishedDate,
                'published_at'    => $publishedIso,
                'created_at'      => $createdIso,
                'tags'            => $tagsList,
                'shelf'           => $shelf,
                'blogger_id'      => $idUri,
                'blogger_type'    => $bloggerType,
                'blogger_status'  => $bloggerStatus,
                'blogger_created' => $bloggerCreated,
                'blogger_filename'=> $bloggerFile,
                'blogger_location'=> $bloggerLoc,
                'blogger_trashed' => $bloggerTrashed,
            ];

            $metaPhp = tlibrary_export_meta_php($meta);

            $pageCanonical = rtrim($TLIBRARY_BASE_URL, '/') . '/tlibrary/' . $slug . '.php';

            $bodyHtml = "<div class=\"tlibrary-entry-body\">\n" . $contentHtml . "\n</div>";

            $php = "<?php\n"
                . "// ============================================================================\n"
                . "// TLibrary Tome — auto-generated from local GOD file\n"
                . "// Slug: {$slug}\n"
                . "// ---------------------------------------------------------------------------\n\n"
                . "\$tlibrary_meta = {$metaPhp};\n\n"
                . "\$page_title       = " . tlibrary_php_string($pageTitle . ' | The Republic') . ";\n"
                . "\$page_canonical   = " . tlibrary_php_string($pageCanonical) . ";\n"
                . "\$page_description = " . tlibrary_php_string($snippet) . ";\n\n"
                . "\$page_og_title       = \$page_title;\n"
                . "\$page_og_description = \$page_description;\n"
                . "\$page_og_url         = \$page_canonical;\n"
                . "\$page_og_image       = " . tlibrary_php_string(rtrim($TLIBRARY_BASE_URL, '/') . '/images/THeart.png') . ";\n\n"
                . "\$hero_title   = 'TLibrary Tome';\n"
                . "\$hero_tagline = '📘 Book · ⚖️ Law · 📜 Chronicle of The Republic';\n\n"
                . "\$console_title = " . tlibrary_php_string($pageTitle) . ";\n\n"
                . "\$console_body_html = <<<'HTML'\n"
                . $bodyHtml . "\n"
                . "HTML;\n\n"
                . "require __DIR__ . '/../tshell.php';\n";

            @file_put_contents($filePath, $php);
            $createdFiles++;
        }

        // Index entry
        $index[] = [
            'id'              => $slug,
            'slug'            => $slug,
            'kind'            => $kind,
            'title'           => $title !== '' ? $title : $slug,
            'published'       => $publishedDate,
            'published_at'    => $publishedIso,
            'created_at'      => $createdIso,
            'tags'            => $tagsList,
            'shelf'           => $shelf,
            'blogger_id'      => $idUri,
            'blogger_type'    => $bloggerType,
            'blogger_status'  => $bloggerStatus,
            'blogger_created' => $bloggerCreated,
            'blogger_filename'=> $bloggerFile,
            'blogger_location'=> $bloggerLoc,
        ];
    }

    // Sort newest first (published_at then published)
    usort($index, static function ($a, $b) {
        $ap = $a['published_at'] ?? '';
        $bp = $b['published_at'] ?? '';
        if ($ap && $bp && $ap !== $bp) {
            return strcmp($bp, $ap);
        }
        $ad = $a['published'] ?? '';
        $bd = $b['published'] ?? '';
        return strcmp($bd, $ad);
    });

    $json = json_encode($index, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    if ($json === false) {
        $outMessage = '❌ Could not encode tlibrary-index.json.';
        return false;
    }

    $bytes = @file_put_contents($TLIBRARY_INDEX_JSON, $json);
    if ($bytes === false) {
        $outMessage = '❌ Could not write tlibrary-index.json. Check file permissions.';
        return false;
    }

    $outMessage = '✅ Synced ' . count($index) . ' LIVE ENTRIES from local GOD file '
        . '(processed ' . $totalEntries . ' feed items; created ' . $createdFiles . ' new PHP tomes).';

    return true;
}

// ---------------------------------------------------------------------------
// 3. PUBLIC SYNC BELL (any citizen)
// ---------------------------------------------------------------------------

$tlibrary_sync_message_html = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['tlibrary_sync'])) {
    $ok = tlibrary_sync_from_feed($syncMessage);
    $safeMessage = htmlspecialchars($syncMessage ?? '', ENT_QUOTES, 'UTF-8');

    $tlibrary_sync_message_html = '
      <div class="tl-sync-banner">'
      . ($ok ? '🔔' : '🌀') . ' '
      . $safeMessage .
      ' <span style="opacity:0.75;">(Tomes live in <code>/tlibrary</code>, index in <code>/tlibrary/admin/data/tlibrary-index.json</code>. GOD file at <code>/feed.atom</code>.)</span>
      </div>';
}

// ---------------------------------------------------------------------------
// 4. PAGE METADATA (TShell)
// ---------------------------------------------------------------------------

$page_title       = 'The Republic | TLibrary';
$page_canonical   = $TLIBRARY_BASE_URL . '/tlibrary.php';
$page_description = 'TLibrary is the forbidden magicka archive of The Republic, collecting Books, Laws, and Chronicles in Mythocratic time.';

$page_og_title       = $page_title;
$page_og_description = $page_description;
$page_og_url         = $page_canonical;
$page_og_image       = $TLIBRARY_BASE_URL . '/images/THeart.png';

$hero_title   = 'TLibrary';
$hero_tagline = '📘 Books · ⚖️ Laws · 📜 Chronicles of The Republic';

$console_title = 'TLibrary — Forbidden Magicka Archive';

// ---------------------------------------------------------------------------
// 5. CONSOLE BODY HTML (contract gate + UI)
// ---------------------------------------------------------------------------

$console_body_html = <<<'HTML'
<style>
  .tlibrary-console {
    max-width: 960px;
    margin: 0 auto;
    padding: 14px 16px 18px;
    border-radius: 20px;
    border: 2px solid rgba(91, 207, 250, 0.6);
    background: linear-gradient(
      180deg,
      rgba(91, 207, 250, 0.20),
      rgba(245, 169, 184, 0.18),
      rgba(255, 255, 255, 0.96)
    );
    color: #1f1022;
    font-size: 0.95rem;
    position: relative;
  }

  .tlibrary-console-main {
    transition: filter 160ms ease, opacity 160ms ease;
  }

  .tlibrary-console:not(.is-unlocked) .tlibrary-console-main {
    opacity: 0.2;
    filter: blur(2px);
    pointer-events: none;
    user-select: none;
  }

  .tlibrary-console.is-unlocked .tlibrary-contract {
    display: none;
  }

  .tl-panel {
    background: rgba(255, 255, 255, 0.96);
    border-radius: 14px;
    border: 1px solid rgba(245, 169, 184, 0.55);
    padding: 10px 12px;
    margin-bottom: 16px;
  }

  .tl-panel-title {
    font-weight: 700;
    margin-bottom: 6px;
    color: #d6247d;
  }

  .tl-sync-banner {
    margin-bottom: 10px;
    padding: 6px 9px;
    border-radius: 10px;
    border: 1px dashed rgba(91, 207, 250, 0.7);
    background: rgba(91, 207, 250, 0.15);
    font-size: 0.8rem;
  }

  .tl-sync-row {
    display: flex;
    flex-wrap: wrap;
    align-items: center;
    gap: 6px;
    margin-bottom: 6px;
    font-size: 0.8rem;
  }

  .tl-sync-form {
    margin: 0;
  }

  .tl-sync-bell-link {
    display: inline-flex;
    align-items: center;
    padding: 4px 10px;
    border-radius: 999px;
    border: 1px solid rgba(91, 207, 250, 0.9);
    background: rgba(255, 255, 255, 0.95);
    text-decoration: none;
    color: #0f3f57;
    font-weight: 600;
    cursor: pointer;
  }

  .tl-sync-note {
    opacity: 0.7;
  }

  .tl-intro-text {
    font-size: 0.88rem;
  }

  .tl-input {
    padding: 8px 10px;
    border-radius: 8px;
    border: 1px solid rgba(91, 207, 250, 0.7);
    background: rgba(255, 255, 255, 0.98);
    color: #1f1022;
    font-size: 0.95rem;
    outline: none;
  }

  .tl-chip {
    border-radius: 999px;
    padding: 4px 10px;
    border: 1px solid rgba(91, 207, 250, 0.4);
    background: rgba(255, 255, 255, 0.9);
    cursor: pointer;
    color: #d6247d;
    font-size: 0.8rem;
  }

  .tl-chip-active {
    background: linear-gradient(90deg, #5bcffa, #f5a9b8);
    border-color: rgba(214, 36, 125, 0.85);
    color: #1f1022;
    font-weight: 600;
  }

  .tl-section-header {
    background: rgba(255, 255, 255, 0.96);
    color: #d6247d;
  }

  .tl-section-body {
    background: rgba(255, 255, 255, 0.98);
  }

  .tl-list-link {
    display: block;
    padding: 6px 7px 7px;
    margin-bottom: 4px;
    border-radius: 7px;
    text-decoration: none;
    color: #1f1022;
    background: rgba(255, 255, 255, 0.96);
    border: 1px solid rgba(245, 169, 184, 0.6);
  }

  .tl-list-link:hover {
    border-color: rgba(91, 207, 250, 0.9);
    box-shadow: 0 0 0 1px rgba(91, 207, 250, 0.4);
  }

  .tl-list-title {
    font-size: 0.9rem;
    font-weight: 600;
    word-break: break-word;
  }

  .tl-list-meta {
    font-size: 0.75rem;
    opacity: 0.75;
    margin-top: 2px;
  }

  /* NEW: full canonical URL under every title */
  .tl-list-url {
    font-size: 0.72rem;
    opacity: 0.85;
    word-break: break-all;
    color: #0f3f57;
    margin-top: 1px;
  }

  .tl-button-pill {
    margin-top: 8px;
    padding: 6px 10px;
    border-radius: 999px;
    border: 1px solid rgba(91, 207, 250, 0.8);
    background: rgba(255, 255, 255, 0.96);
    cursor: pointer;
    font-size: 0.8rem;
    color: #0f3f57;
  }

  .tl-explore-grid {
    display: flex;
    flex-wrap: wrap;
    gap: 6px;
    margin-bottom: 6px;
  }

  .tl-explore-target {
    font-size: 0.82rem;
  }

  .tl-footer-signature {
    margin-top: 10px;
    font-size: 0.75rem;
    opacity: 0.8;
    text-align: right;
  }

  /* Contract (18+ forbidden magicka gate) */
  .tlibrary-contract{
    margin:0.2rem 0 0.9rem;
    padding:0.75rem 1rem;
    border-radius:18px;
    border:1px solid rgba(148,163,184,0.9);
    background:radial-gradient(circle at top,#fef3c7,#fee2f2,#eff6ff);
    box-shadow:0 8px 20px rgba(15,23,42,0.18);
    font-size:0.8rem;
  }
  .tlibrary-contract-title{
    font-size:0.85rem;
    text-transform:uppercase;
    letter-spacing:0.18em;
    margin-bottom:0.35rem;
  }
  .tlibrary-contract-body{
    margin:0 0 0.35rem;
  }
  .tlibrary-contract-clauses{
    margin:0.4rem 0 0.4rem;
    display:flex;
    flex-direction:column;
    gap:0.3rem;
  }
  .tlibrary-contract-clause{
    display:flex;
    align-items:flex-start;
    gap:0.4rem;
    font-size:0.78rem;
  }
  .tlibrary-contract-clause input{
    margin-top:0.12rem;
  }
  .tlibrary-contract-actions{
    margin-top:0.45rem;
    display:flex;
    justify-content:flex-start;
  }
  .tlibrary-contract-error{
    margin-top:0.3rem;
    font-size:0.75rem;
    color:#b91c1c;
  }
  .tlibrary-contract-warning{
    margin-top:0.3rem;
    font-size:0.75rem;
    opacity:0.85;
  }

  .tl-select {
    padding: 5px 8px;
    border-radius: 8px;
    border: 1px solid rgba(91,207,250,0.7);
    background: rgba(255,255,255,0.98);
    font-size: 0.8rem;
  }

  /* NEW: sort controls per section */
  .tl-sort-row {
    display:flex;
    justify-content:flex-end;
    align-items:center;
    gap:4px;
    font-size:0.72rem;
    margin-bottom:4px;
    opacity:0.85;
  }
  .tl-sort-label {
    opacity:0.7;
  }
  .tl-sort-button {
    border-radius:999px;
    padding:2px 8px;
    border:1px solid rgba(91,207,250,0.6);
    background:rgba(255,255,255,0.95);
    cursor:pointer;
    font-size:0.72rem;
    color:#0f3f57;
  }
  .tl-sort-button-active {
    background:linear-gradient(90deg,#5bcffa,#f5a9b8);
    border-color:rgba(214,36,125,0.85);
    color:#1f1022;
    font-weight:600;
  }
</style>

<div class="tlibrary-console" id="tlibraryConsoleRoot">

  <!-- Soul pact gate -->
  <section class="tlibrary-contract" id="tlibraryContract">
    <div class="tlibrary-contract-title">
      🕯️ TLibrary Gate — Forbidden Magicka Shelf
    </div>
    <p class="tlibrary-contract-body">
      Outside authorities would mark this archive as <strong>18+ forbidden material</strong> if they understood it,
      not because it is vulgar, but because it hands ordinary people the tools of sovereignty.
    </p>
    <p class="tlibrary-contract-body">
      If you are not yet an adult where you live, or it is unlawful for you to read material labelled “18+”,
      The Republic’s verdict is simple: <strong>turn back now</strong>. Close this page.
    </p>
    <p class="tlibrary-contract-body">
      If you stay, you accept this as a <strong>magicka school</strong>, not a wellness product. These tomes
      may confront you more than they comfort you.
    </p>

    <div class="tlibrary-contract-clauses">
      <label class="tlibrary-contract-clause">
        <input type="checkbox" id="tlibraryContractAge">
        <span>I am at least 18 years old (or legal adulthood where I live) and it is lawful for me to read 18+ archives.</span>
      </label>
      <label class="tlibrary-contract-clause">
        <input type="checkbox" id="tlibraryContractConsent">
        <span>I understand TLibrary is a magicka school; these books may confront me more than they comfort me.</span>
      </label>
      <label class="tlibrary-contract-clause">
        <input type="checkbox" id="tlibraryContractQueens">
        <span>Inside this shelf I align with Republic law: trans women are Queens, consent is sacred, and every page is a gift.</span>
      </label>
    </div>

    <div class="tlibrary-contract-actions">
      <button type="button" id="tlibraryContractEnter" class="tl-button-pill">
        ⚡ Swear the Pact & Open TLibrary
      </button>
    </div>
    <div id="tlibraryContractError" class="tlibrary-contract-error"></div>
    <div class="tlibrary-contract-warning">
      If you are under the legal age for adult material where you live, do not click the button. Close this page instead.
    </div>
  </section>

  <div class="tlibrary-console-main" id="tlibraryConsoleMain">

    [[TLIB_SYNC_STATUS]]

    <div class="tl-sync-row">
      <form method="post" class="tl-sync-form">
        <button type="submit"
                name="tlibrary_sync"
                value="1"
                class="tl-sync-bell-link">
          🔔 Sync Bell • Call the tomes into orbit
        </button>
      </form>
      <span class="tl-sync-note">
        Any citizen may ring this bell. It reads the local GOD file at <code>/feed.atom</code>,
        writes PHP tomes into <code>/tlibrary</code>, and refreshes the JSON catalogue.
      </span>
    </div>

    <!-- SHORT INTRO FOR CITIZENS -->
    <section class="tl-panel">
      <div class="tl-panel-title">🕯️ Welcome to TLibrary</div>
      <div class="tl-intro-text">
        This console is the forbidden reading room of The Republic. Every 📘 Book, ⚖️ Law, and 📜 Chronicle
        lives here in Mythocratic time. The outside world calls it “18+”; inside, we call it curriculum.
      </div>
    </section>

    <!-- EXPLORE TOOLS -->
    <section class="tl-panel">
      <div class="tl-panel-title">🧭 Explore TLibrary</div>
      <div class="tl-explore-grid">
        <button type="button" id="tl-explore-random-book" class="tl-button-pill">🎲 Random Book</button>
        <button type="button" id="tl-explore-foundation" class="tl-button-pill">🏛️ Foundational Shelf</button>
        <button type="button" id="tl-explore-latest" class="tl-button-pill">🕰 Latest Chronicle</button>
      </div>
      <div id="tl-explore-target" class="tl-explore-target">
        Choose an explorer above, or scroll down to browse by kind.
      </div>
    </section>

    <!-- TAG EXPLORERS -->
    <section class="tl-panel">
      <div class="tl-panel-title">🔖 Tag Explorers</div>
      <div id="tl-tag-explorer-chips" class="tl-explore-grid"></div>
      <div id="tl-tag-explorer-note" class="tl-explore-target">
        Click a tag to jump the catalogue to that shelf. Tag + kind + year all stack with the search box.
      </div>
    </section>

    <!-- SEARCH PANEL -->
    <section class="tl-panel">
      <div class="tl-panel-title">🔍 Seek within TLibrary</div>
      <div style="margin-bottom: 6px; font-size: 0.82rem; opacity: 0.8;">
        Search titles only across 📘 Books, ⚖️ Laws, and 📜 Chronicles. Use the dropdowns to filter by tag and year.
      </div>

      <div style="display:flex;flex-direction:column;gap:8px;">
        <input
          id="tl-search-input"
          class="tl-input"
          type="text"
          placeholder="Type a word or phrase from a title..."
        />

        <div id="tl-search-kind-filters" style="display:flex;flex-wrap:wrap;gap:6px;font-size:0.8rem;">
          <button type="button" data-kind="all" class="tl-chip tl-chip-active">
            ✨ All
          </button>
          <button type="button" data-kind="book" class="tl-chip">
            📘 Books
          </button>
          <button type="button" data-kind="law" class="tl-chip">
            ⚖️ Laws
          </button>
          <button type="button" data-kind="chronicle" class="tl-chip">
            📜 Chronicles
          </button>
        </div>

        <div style="display:flex;flex-wrap:wrap;gap:6px;font-size:0.8rem;">
          <label>
            <span style="font-size:0.75rem;opacity:0.8;">Tag</span><br>
            <select id="tl-search-tag" class="tl-select"></select>
          </label>
          <label>
            <span style="font-size:0.75rem;opacity:0.8;">Year</span><br>
            <select id="tl-search-year" class="tl-select"></select>
          </label>
        </div>

        <div id="tl-search-status" style="font-size:0.8rem;opacity:0.8;"></div>

        <div id="tl-search-results" style="margin-top:4px;max-height:260px;overflow-y:auto;border-radius:8px;"></div>
      </div>
    </section>

    <!-- BOOKS SECTION -->
    <section class="tl-panel tl-section" data-kind="book" style="margin-bottom: 14px;">
      <button
        type="button"
        class="tl-section-header"
        data-kind="book"
        style="width:100%;text-align:left;padding:8px 10px;border:none;cursor:pointer;display:flex;align-items:center;justify-content:space-between;font-size:0.98rem;font-weight:600;"
      >
        <span>📘 Books of The Republic <span id="tl-count-book" style="opacity:0.7;font-weight:400;"></span></span>
        <span class="tl-section-chevron">▾</span>
      </button>
      <div id="tl-body-book" class="tl-section-body" style="display:block;padding:8px 9px;">
        <div class="tl-sort-row" data-kind="book">
          <span class="tl-sort-label">Order:</span>
          <button type="button" class="tl-sort-button tl-sort-button-active" data-order="newest">
            🕰 Newest → Oldest
          </button>
          <button type="button" class="tl-sort-button" data-order="oldest">
            🌱 Oldest → Newest
          </button>
        </div>
        <div id="tl-list-book"></div>
        <button type="button" id="tl-more-book" class="tl-button-pill">
          ⏭️ Load 20 more books
        </button>
      </div>
    </section>

    <!-- LAWS SECTION -->
    <section class="tl-panel tl-section" data-kind="law" style="margin-bottom: 14px;">
      <button
        type="button"
        class="tl-section-header"
        data-kind="law"
        style="width:100%;text-align:left;padding:8px 10px;border:none;cursor:pointer;display:flex;align-items:center;justify-content:space-between;font-size:0.98rem;font-weight:600;"
      >
        <span>⚖️ Laws of The Republic <span id="tl-count-law" style="opacity:0.7;font-weight:400;"></span></span>
        <span class="tl-section-chevron">▾</span>
      </button>
      <div id="tl-body-law" class="tl-section-body" style="display:none;padding:8px 9px;">
        <div class="tl-sort-row" data-kind="law">
          <span class="tl-sort-label">Order:</span>
          <button type="button" class="tl-sort-button tl-sort-button-active" data-order="newest">
            🕰 Newest → Oldest
          </button>
          <button type="button" class="tl-sort-button" data-order="oldest">
            🌱 Oldest → Newest
          </button>
        </div>
        <div id="tl-list-law"></div>
        <button type="button" id="tl-more-law" class="tl-button-pill">
          ⏭️ Load 20 more laws
        </button>
      </div>
    </section>

    <!-- CHRONICLES SECTION -->
    <section class="tl-panel tl-section" data-kind="chronicle" style="margin-bottom: 6px;">
      <button
        type="button"
        class="tl-section-header"
        data-kind="chronicle"
        style="width:100%;text-align:left;padding:8px 10px;border:none;cursor:pointer;display:flex;align-items:center;justify-content:space-between;font-size:0.98rem;font-weight:600;"
      >
        <span>📜 Chronicles of The Republic <span id="tl-count-chronicle" style="opacity:0.7;font-weight:400;"></span></span>
        <span class="tl-section-chevron">▾</span>
      </button>
      <div id="tl-body-chronicle" class="tl-section-body" style="display:none;padding:8px 9px;">
        <div class="tl-sort-row" data-kind="chronicle">
          <span class="tl-sort-label">Order:</span>
          <button type="button" class="tl-sort-button tl-sort-button-active" data-order="newest">
            🕰 Newest → Oldest
          </button>
          <button type="button" class="tl-sort-button" data-order="oldest">
            🌱 Oldest → Newest
          </button>
        </div>
        <div id="tl-list-chronicle"></div>
        <button type="button" id="tl-more-chronicle" class="tl-button-pill">
          ⏭️ Load 20 more chronicles
        </button>
      </div>
    </section>

    <div class="tl-footer-signature">
      TLibrary engine & sync bell seeded by Codey, Republic Systems Programmer 💻👑
    </div>

  </div><!-- /.tlibrary-console-main -->
</div><!-- /.tlibrary-console -->

<script>
// ==========================================================================
// TLibrary client — title search, tag/year filters, explorers, sections
// Powered by /tlibrary/admin/data/tlibrary-index.json
// Authored by Codey, Republic Systems Programmer 💻👑
// ==========================================================================

(function() {
  'use strict';

  const DATA_URL  = '/tlibrary/admin/data/tlibrary-index.json';
  const PAGE_SIZE = 20;

  const state = {
    index: [],
    groups: {
      book: [],
      law: [],
      chronicle: []
    },
    offsets: {
      book: PAGE_SIZE,
      law: PAGE_SIZE,
      chronicle: PAGE_SIZE
    },
    searchKind: 'all',
    searchQuery: '',
    tags: [],
    years: [],
    sortOrder: {
      book: 'newest',
      law: 'newest',
      chronicle: 'newest'
    }
  };

  // Contract gate
  (function initContractGate() {
    const root     = document.getElementById('tlibraryConsoleRoot');
    const enterBtn = document.getElementById('tlibraryContractEnter');
    const ageEl    = document.getElementById('tlibraryContractAge');
    const consentEl= document.getElementById('tlibraryContractConsent');
    const queensEl = document.getElementById('tlibraryContractQueens');
    const errEl    = document.getElementById('tlibraryContractError');

    if (!root || !enterBtn || !ageEl || !consentEl || !queensEl) return;

    function unlock() {
      root.classList.add('is-unlocked');
    }

    enterBtn.addEventListener('click', function () {
      if (!ageEl.checked || !consentEl.checked || !queensEl.checked) {
        if (errEl) {
          errEl.textContent =
            '🌀🛰️⟦ᛉᛞᛟ⟧ Ｑ：ＡＲＥ　ＹＯＵ　ＴＲＹＩＮＧ　ＴＯ　ＳＫＩＰ　ＴＨＥ　ＰＡＣＴ？ 🔴⚠️⛔ SYSTEM: All three clauses must be true before the shelves unlock. 🛰️⟦ᛉᛞᛟ⟧🌀';
        }
        return;
      }
      if (errEl) errEl.textContent = '';
      unlock();
    });
  })();

  const searchInput    = document.getElementById('tl-search-input');
  const searchStatus   = document.getElementById('tl-search-status');
  const searchResults  = document.getElementById('tl-search-results');
  const filterContainer= document.getElementById('tl-search-kind-filters');
  const tagSelect      = document.getElementById('tl-search-tag');
  const yearSelect     = document.getElementById('tl-search-year');
  const tagExplorer    = document.getElementById('tl-tag-explorer-chips');
  const tagExplorerNote= document.getElementById('tl-tag-explorer-note');

  function formatMythocraticDate(ymd) {
    if (!ymd || typeof ymd !== 'string') return '';

    const parts = ymd.split('-');
    if (parts.length < 3) return ymd;

    const year = parseInt(parts[0], 10);
    const monthNum = parseInt(parts[1], 10);
    const day = parseInt(parts[2], 10);

    if (isNaN(year) || isNaN(monthNum) || isNaN(day)) return ymd;

    const months = [
      'January','February','March','April','May','June',
      'July','August','September','October','November','December'
    ];
    const monthName = months[Math.max(0, Math.min(11, monthNum - 1))];
    const ceYear = year;

    let mythYearStr;
    if (year >= 2025) {
      const mcYear = year - 2025 + 1;
      const mcStr = String(mcYear).padStart(4, '0') + ' MC';
      mythYearStr = mcStr;
    } else {
      const pmcOffset = 2025 - year;
      const pmcStr = '-' + String(pmcOffset).padStart(4, '0') + ' PMC';
      mythYearStr = pmcStr;
    }

    return monthName + ' ' + day + ', ' + mythYearStr + ' (' + ceYear + ' CE)';
  }

  function byPublishedDesc(a, b) {
    const ap = a.published_at || '';
    const bp = b.published_at || '';
    if (ap && bp && ap !== bp) {
      return bp.localeCompare(ap);
    }
    const ad = a.published || '';
    const bd = b.published || '';
    return bd.localeCompare(ad);
  }

  function populateTagYearDropdowns() {
    if (!tagSelect || !yearSelect) return;

    // Tag dropdown
    tagSelect.innerHTML = '';
    let opt = document.createElement('option');
    opt.value = '';
    opt.textContent = 'All tags';
    tagSelect.appendChild(opt);
    state.tags.forEach(function(tag) {
      const o = document.createElement('option');
      o.value = tag;
      o.textContent = tag;
      tagSelect.appendChild(o);
    });

    // Year dropdown
    yearSelect.innerHTML = '';
    let oy = document.createElement('option');
    oy.value = '';
    oy.textContent = 'All years';
    yearSelect.appendChild(oy);
    state.years.forEach(function(year) {
      const y = document.createElement('option');
      y.value = year;
      y.textContent = year;
      yearSelect.appendChild(y);
    });
  }

  function renderTagExplorers() {
    if (!tagExplorer) return;

    tagExplorer.innerHTML = '';

    if (!state.tags.length) {
      if (tagExplorerNote) {
        tagExplorerNote.textContent = 'No tags detected yet. Ring the Sync Bell after filling the GOD file.';
      }
      return;
    }

    state.tags.slice(0, 40).forEach(function(tag) {
      const btn = document.createElement('button');
      btn.type = 'button';
      btn.className = 'tl-chip';
      btn.textContent = tag;
      btn.addEventListener('click', function() {
        if (tagSelect) {
          tagSelect.value = tag;
        }
        if (tagExplorerNote) {
          tagExplorerNote.textContent = '🔖 Tag filter active: "' + tag + '". Use kind/year/search to refine.';
        }
        performSearch();
        const searchPanel = document.getElementById('tl-search-input');
        if (searchPanel && searchPanel.scrollIntoView) {
          searchPanel.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
      });
      tagExplorer.appendChild(btn);
    });
  }

  function renderSection(kind) {
    const listEl = document.getElementById('tl-list-' + kind);
    const moreBtn= document.getElementById('tl-more-' + kind);
    if (!listEl || !state.groups[kind]) return;

    const entries = state.groups[kind];
    const max   = state.offsets[kind] || PAGE_SIZE;

    const order = (state.sortOrder && state.sortOrder[kind]) || 'newest';
    const ordered = entries.slice();
    if (order === 'oldest') {
      ordered.reverse();
    }

    listEl.innerHTML = '';

    const slice = ordered.slice(0, max);
    slice.forEach(function(entry) {
      const a = document.createElement('a');
      a.href = '/tlibrary/' + entry.slug + '.php';
      a.className = 'tl-list-link';

      const titleDiv = document.createElement('div');
      titleDiv.className = 'tl-list-title';
      titleDiv.textContent = entry.title || entry.slug;

      // NEW: full canonical URL below the title
      const urlDiv = document.createElement('div');
      urlDiv.className = 'tl-list-url';
      urlDiv.textContent = 'https://trepublic.net/tlibrary/' + entry.slug + '.php';

      const metaDiv = document.createElement('div');
      metaDiv.className = 'tl-list-meta';

      const mythDate = entry.published ? formatMythocraticDate(entry.published) : '';
      const tagLine = (Array.isArray(entry.tags) && entry.tags.length)
        ? entry.tags.join(', ')
        : '';

      let kindEmoji = '📜';
      if (entry.kind === 'book') kindEmoji = '📘';
      else if (entry.kind === 'law') kindEmoji = '⚖️';

      const bits = [];
      if (mythDate) bits.push(mythDate);
      if (tagLine) bits.push(tagLine);

      metaDiv.textContent = kindEmoji + ' ' + bits.join(' • ');

      a.appendChild(titleDiv);
      a.appendChild(urlDiv);   // NEW
      a.appendChild(metaDiv);
      listEl.appendChild(a);
    });

    if (moreBtn) {
      moreBtn.style.display = ordered.length > max ? 'inline-flex' : 'none';
    }
  }

  function wireLoadMore(kind) {
    const btn = document.getElementById('tl-more-' + kind);
    if (!btn) return;
    btn.addEventListener('click', function() {
      state.offsets[kind] = (state.offsets[kind] || PAGE_SIZE) + PAGE_SIZE;
      renderSection(kind);
    });
  }

  function wireAccordions() {
    const headers = document.querySelectorAll('.tl-section-header');
    headers.forEach(function(header) {
      header.addEventListener('click', function() {
        const kind = header.getAttribute('data-kind');
        const body = document.getElementById('tl-body-' + kind);
        const chev = header.querySelector('.tl-section-chevron');
        if (!body || !chev) return;
        const isHidden = body.style.display === 'none';
        body.style.display = isHidden ? 'block' : 'none';
        chev.textContent = isHidden ? '▾' : '▸';
      });
    });
  }

  function applySort(kind, order) {
    if (!state.sortOrder) state.sortOrder = {};
    state.sortOrder[kind] = (order === 'oldest') ? 'oldest' : 'newest';
    state.offsets[kind] = PAGE_SIZE;
    renderSection(kind);

    const row = document.querySelector('.tl-sort-row[data-kind="' + kind + '"]');
    if (row) {
      const buttons = row.querySelectorAll('.tl-sort-button');
      buttons.forEach(function(btn) {
        const btnOrder = btn.getAttribute('data-order') || 'newest';
        if (btnOrder === state.sortOrder[kind]) {
          btn.classList.add('tl-sort-button-active');
        } else {
          btn.classList.remove('tl-sort-button-active');
        }
      });
    }
  }

  function wireSortButtons() {
    const rows = document.querySelectorAll('.tl-sort-row');
    rows.forEach(function(row) {
      const kind = row.getAttribute('data-kind');
      if (!kind) return;
      row.addEventListener('click', function(e) {
        const btn = e.target.closest('.tl-sort-button');
        if (!btn) return;
        const order = btn.getAttribute('data-order') || 'newest';
        applySort(kind, order);
      });
    });
  }

  function performSearch() {
    if (!searchStatus || !searchResults) return;

    const qRaw = searchInput ? searchInput.value || '' : '';
    const q = qRaw.trim().toLowerCase();
    const kind = state.searchKind;
    const tag = tagSelect ? (tagSelect.value || '') : '';
    const year = yearSelect ? (yearSelect.value || '') : '';

    const hasFilter = q || tag || year || kind !== 'all';

    if (!hasFilter) {
      searchResults.innerHTML = '';
      searchStatus.textContent = 'Type to search titles, or browse by kind below.';
      return;
    }

    const matches = state.index.filter(function(entry) {
      if (!entry) return false;
      if (kind !== 'all' && entry.kind !== kind) return false;

      if (tag) {
        if (!Array.isArray(entry.tags) || entry.tags.indexOf(tag) === -1) {
          return false;
        }
      }

      if (year) {
        if (!entry.published || String(entry.published).slice(0, 4) !== year) {
          return false;
        }
      }

      if (q) {
        const t = (entry.title || '').toLowerCase();
        return t.indexOf(q) !== -1;
      }

      return true;
    });

    searchResults.innerHTML = '';

    if (!matches.length) {
      searchStatus.textContent = 'No tomes matched that query / filter.';
      return;
    }

    searchStatus.textContent = 'Showing ' + matches.length + ' matching tome(s).';

    matches.slice(0, 160).forEach(function(entry) {
      const a = document.createElement('a');
      a.href = '/tlibrary/' + entry.slug + '.php';
      a.className = 'tl-list-link';

      const titleDiv = document.createElement('div');
      titleDiv.className = 'tl-list-title';
      titleDiv.textContent = entry.title || entry.slug;

      // NEW: full canonical URL in search results
      const urlDiv = document.createElement('div');
      urlDiv.className = 'tl-list-url';
      urlDiv.textContent = 'https://trepublic.net/tlibrary/' + entry.slug + '.php';

      const metaDiv = document.createElement('div');
      metaDiv.className = 'tl-list-meta';

      const mythDate = entry.published ? formatMythocraticDate(entry.published) : '';
      const tagLine = (Array.isArray(entry.tags) && entry.tags.length)
        ? entry.tags.join(', ')
        : '';

      let kindEmoji = '📜';
      if (entry.kind === 'book') kindEmoji = '📘';
      else if (entry.kind === 'law') kindEmoji = '⚖️';

      const bits = [];
      if (mythDate) bits.push(mythDate);
      if (tagLine) bits.push(tagLine);

      metaDiv.textContent = kindEmoji + ' ' + bits.join(' • ');

      a.appendChild(titleDiv);
      a.appendChild(urlDiv);   // NEW
      a.appendChild(metaDiv);
      searchResults.appendChild(a);
    });
  }

  function wireSearch() {
    if (!filterContainer) return;

    filterContainer.addEventListener('click', function(e) {
      const btn = e.target.closest('button[data-kind]');
      if (!btn) return;
      const kind = btn.getAttribute('data-kind') || 'all';
      state.searchKind = kind;

      const chips = filterContainer.querySelectorAll('.tl-chip');
      chips.forEach(function(chip) {
        chip.classList.remove('tl-chip-active');
      });
      btn.classList.add('tl-chip-active');
      performSearch();
    });

    if (searchInput) {
      searchInput.addEventListener('input', function() {
        performSearch();
      });
    }

    if (tagSelect) {
      tagSelect.addEventListener('change', performSearch);
    }
    if (yearSelect) {
      yearSelect.addEventListener('change', performSearch);
    }
  }

  function pickRandom(arr) {
    if (!arr || !arr.length) return null;
    const idx = Math.floor(Math.random() * arr.length);
    return arr[idx];
  }

  function wireExploreButtons() {
    const randomBtn = document.getElementById('tl-explore-random-book');
    const foundationBtn = document.getElementById('tl-explore-foundation');
    const latestBtn = document.getElementById('tl-explore-latest');
    const target = document.getElementById('tl-explore-target');

    if (randomBtn) {
      randomBtn.addEventListener('click', function() {
        const entry = pickRandom(state.groups.book.length ? state.groups.book : state.index);
        if (!entry || !target) return;
        target.innerHTML =
          '🎲 Random tome: <a href="/tlibrary/' + entry.slug + '.php">' +
          (entry.title || entry.slug) +
          '</a>';
      });
    }

    if (foundationBtn) {
      foundationBtn.addEventListener('click', function() {
        if (!target) return;
        const shelf = state.index.filter(function(e) {
          return e && e.shelf === 'Foundational 11';
        });
        if (!shelf.length) {
          target.textContent = '🏛️ No Foundational 11 tomes detected yet. Ring the Sync Bell after filling the GOD file.';
          return;
        }
        const lines = shelf.slice(0, 20).map(function(e) {
          return '<li><a href="/tlibrary/' + e.slug + '.php">' +
            (e.title || e.slug) + '</a></li>';
        }).join('');
        target.innerHTML =
          '🏛️ Foundational 11 shelf (' + shelf.length + '):<ul style="margin:4px 0 0 1em;padding:0;">' +
          lines + '</ul>';
      });
    }

    if (latestBtn) {
      latestBtn.addEventListener('click', function() {
        if (!target) return;
        if (!state.index.length) {
          target.textContent = '🕰 No tomes yet. Ring the Sync Bell to import the GOD file.';
          return;
        }
        const latest = state.index.slice().sort(byPublishedDesc)[0];
        target.innerHTML =
          '🕰 Latest chronicle: <a href="/tlibrary/' + latest.slug + '.php">' +
          (latest.title || latest.slug) + '</a>';
      });
    }
  }

  function init() {
    if (!searchStatus || !searchResults) {
      return;
    }

    searchStatus.textContent = 'Loading TLibrary index…';

    fetch(DATA_URL, { cache: 'no-cache' })
      .then(function(res) {
        if (!res.ok) {
          throw new Error('HTTP ' + res.status);
        }
        return res.json();
      })
      .then(function(data) {
        if (!Array.isArray(data)) {
          searchStatus.textContent = 'Error: index data is not an array.';
          return;
        }

        state.index = data.slice();

        const books = [];
        const laws = [];
        const chronicles = [];

        const tagCount = new Map();
        const yearSet = new Set();

        state.index.forEach(function(entry) {
          if (!entry) return;

          const kind = entry.kind || 'chronicle';
          if (kind === 'book')      books.push(entry);
          else if (kind === 'law')  laws.push(entry);
          else                      chronicles.push(entry);

          if (Array.isArray(entry.tags)) {
            entry.tags.forEach(function(t) {
              if (!t) return;
              const current = tagCount.get(t) || 0;
              tagCount.set(t, current + 1);
            });
          }

          if (entry.published) {
            const parts = String(entry.published).split('-');
            if (parts.length >= 1) {
              yearSet.add(parts[0]);
            }
          }
        });

        books.sort(byPublishedDesc);
        laws.sort(byPublishedDesc);
        chronicles.sort(byPublishedDesc);

        state.groups.book = books;
        state.groups.law = laws;
        state.groups.chronicle = chronicles;

        state.tags = Array.from(tagCount.entries())
          .sort(function(a, b) { return b[1] - a[1]; })
          .map(function(pair) { return pair[0]; });

        state.years = Array.from(yearSet).sort().reverse();

        populateTagYearDropdowns();
        renderTagExplorers();

        const countBook = document.getElementById('tl-count-book');
        const countLaw = document.getElementById('tl-count-law');
        const countChronicle = document.getElementById('tl-count-chronicle');
        if (countBook) countBook.textContent = ' (' + books.length + ')';
        if (countLaw) countLaw.textContent = ' (' + laws.length + ')';
        if (countChronicle) countChronicle.textContent = ' (' + chronicles.length + ')';

        renderSection('book');
        renderSection('law');
        renderSection('chronicle');

        wireLoadMore('book');
        wireLoadMore('law');
        wireLoadMore('chronicle');
        wireAccordions();
        wireSearch();
        wireExploreButtons();
        wireSortButtons();

        searchStatus.textContent = 'Ready. Type to search titles, or filter by tag/year, or browse sections below.';
      })
      .catch(function(err) {
        console.error('TLibrary index load error:', err);
        searchStatus.textContent = 'Error loading TLibrary index (' + err.message + '). Ring the Sync Bell, then reload.';
      });
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
</script>
HTML;

// Inject sync banner if present
if ($tlibrary_sync_message_html !== '') {
    $console_body_html = str_replace('[[TLIB_SYNC_STATUS]]', $tlibrary_sync_message_html, $console_body_html);
} else {
    $console_body_html = str_replace('[[TLIB_SYNC_STATUS]]', '', $console_body_html);
}

require __DIR__ . '/tshell.php';
