<?php
// TComic — 📚 Comic grid viewer inside TShell console

$page_id    = 'tcomic';
$page_title = 'TComic — Wendell\'s Diary';

// Tell TShell what this console is
$console_title = 'TComic — Grid Viewer';

// Basic viewer config
$totalPages     = 368;
$fullFolderWeb  = '/tcomic/';          // full-size PNGs like 001.png
$thumbFolderWeb = '/tcomic/thumbs/';   // JPEG thumbs like 001.jpg
$padLength      = 3;                   // 001, 002, ...
$fullExt        = '.png';
$thumbExt       = '.jpg';

ob_start();
?>
<div class="tcomic-console">
  <div class="tcomic-header">
    <div class="tcomic-header-main">
      <div class="tcomic-eyebrow">📚 TRepublic • 🎨 TComic</div>
      <h2 class="tcomic-title">Wendell's Diary — Comic Grid Viewer 💭</h2>
      <p class="tcomic-subtitle">
	   🌙 Each panel is a little prophecy shard. Keep wandering and the diary will unfurl itself.<br>
      </p>
    </div>
    <div class="tcomic-header-actions">
      <a class="tcomic-btn tcomic-btn-secondary" href="/tcomic/tcomic.zip" download>
        ⬇️ Download full TComic (.zip)
      </a>
    </div>
  </div>

  <div class="tcomic-status-row">
    <span id="tcomicStatus" class="tcomic-status">
      🎬 Preparing the strip…
    </span>
    <span id="tcomicError" class="tcomic-error"></span>
  </div>

  <div id="tcomicGrid" class="tcomic-grid"></div>

  <div class="tcomic-load-more-row">
    <button id="tcomicLoadMore" class="tcomic-btn tcomic-btn-primary" type="button">
      ➕ Load more pages
    </button>
  </div>

  <!-- Lightbox viewer -->
  <div id="tcomicLightbox" class="tcomic-lightbox" aria-hidden="true">
    <div class="tcomic-lightbox-backdrop"></div>
    <div class="tcomic-lightbox-inner" role="dialog" aria-modal="true" aria-labelledby="tcomicLightboxCaption">
      <button id="tcomicLightboxClose" class="tcomic-lightbox-close" type="button" aria-label="Close viewer">
        ✖
      </button>
      <img id="tcomicLightboxImage" class="tcomic-lightbox-image" src="" alt="Full comic page">
      <div id="tcomicLightboxCaption" class="tcomic-lightbox-caption"></div>
      <a id="tcomicLightboxDownload" class="tcomic-btn tcomic-btn-secondary tcomic-lightbox-download" href="#" download>
        ⬇️ Download this page
      </a>
    </div>
  </div>
</div>

<style>
  .tcomic-console {
    max-width: 1320px;
    margin: 0 auto;
    padding: 0.75rem 0.75rem 1.25rem;
  }

  @media (min-width: 768px) {
    .tcomic-console {
      padding: 1rem 1.25rem 1.5rem;
    }
  }

  .tcomic-header {
    display: flex;
    flex-direction: column;
    gap: 0.6rem;
    margin-bottom: 0.6rem;
    padding: 0.75rem 0.9rem;
    border-radius: 18px;
    background: radial-gradient(circle at top left, rgba(168,85,247,0.22), rgba(15,23,42,0.92));
    box-shadow: 0 16px 32px rgba(0,0,0,0.6);
    border: 1px solid rgba(148,163,184,0.6);
  }

  @media (min-width: 768px) {
    .tcomic-header {
      flex-direction: row;
      justify-content: space-between;
      align-items: center;
      gap: 1.25rem;
    }
  }

  .tcomic-eyebrow {
    font-size: 0.7rem;
    text-transform: uppercase;
    letter-spacing: 0.18em;
    opacity: 0.8;
    margin-bottom: 0.15rem;
  }

  .tcomic-title {
    margin: 0;
    font-size: 1.15rem;
    letter-spacing: 0.06em;
    text-transform: uppercase;
  }

  .tcomic-subtitle {
    margin: 0.25rem 0 0;
    font-size: 0.8rem;
    opacity: 0.9;
  }

  .tcomic-header-actions {
    display: flex;
    flex-wrap: wrap;
    gap: 0.4rem;
    align-items: center;
    justify-content: flex-start;
  }

  .tcomic-btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.35rem;
    border-radius: 999px;
    border: 1px solid rgba(167,139,250,0.85);
    font-size: 0.78rem;
    padding: 0.35rem 0.95rem;
    text-decoration: none;
    white-space: nowrap;
    cursor: pointer;
    box-shadow: 0 10px 24px rgba(0,0,0,0.7);
    transition: transform 120ms, box-shadow 120ms, filter 120ms, background 120ms;
    background: transparent;
  }

  .tcomic-btn-primary {
    background: radial-gradient(circle at top left, #c084fc, #6d28d9);
    color: #f9fafb;
  }

  .tcomic-btn-secondary {
    background: rgba(15,23,42,0.96);
    color: #e5e7eb;
  }

  .tcomic-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 14px 30px rgba(0,0,0,0.8);
    filter: brightness(1.05);
  }

  .tcomic-btn:disabled {
    opacity: 0.45;
    cursor: default;
    box-shadow: none;
    transform: none;
    filter: grayscale(0.3);
  }

  .tcomic-status-row {
    display: flex;
    flex-wrap: wrap;
    justify-content: space-between;
    gap: 0.35rem;
    align-items: center;
    margin: 0.6rem 0 0.4rem;
    font-size: 0.78rem;
  }

  .tcomic-status {
    opacity: 0.9;
  }

  .tcomic-error {
    color: #fecaca;
    min-height: 1.2em;
  }

  .tcomic-grid {
    display: grid;
    grid-template-columns: repeat(2, minmax(0, 1fr));
    gap: 0.6rem;
    margin-bottom: 0.7rem;
  }

  @media (min-width: 640px) {
    .tcomic-grid {
      grid-template-columns: repeat(3, minmax(0, 1fr));
    }
  }

  @media (min-width: 960px) {
    .tcomic-grid {
      grid-template-columns: repeat(4, minmax(0, 1fr));
    }
  }

  .tcomic-card {
    background: radial-gradient(circle at top left, rgba(79,70,229,0.18), rgba(15,23,42,0.96));
    border-radius: 16px;
    padding: 0.4rem;
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
    box-shadow: 0 14px 30px rgba(0,0,0,0.6);
    border: 1px solid rgba(148,163,184,0.6);
    overflow: hidden;
    text-decoration: none;
    color: inherit;
    cursor: zoom-in;
  }

  .tcomic-card img {
    width: 100%;
    height: auto;
    display: block;
    border-radius: 12px;
    background: #020617;
    object-fit: contain;
  }

  .tcomic-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 0.7rem;
    padding: 0 0.1rem 0.1rem;
    opacity: 0.95;
  }

  .tcomic-meta-left::before {
    content: "📄 ";
  }

  .tcomic-badge {
    padding: 0.1rem 0.5rem;
    border-radius: 999px;
    border: 1px solid rgba(167,139,250,0.7);
    font-size: 0.65rem;
    background: rgba(15,23,42,0.96);
    white-space: nowrap;
  }

  .tcomic-load-more-row {
    display: flex;
    justify-content: center;
    margin-bottom: 0.75rem;
  }

  .tcomic-hint {
    text-align: center;
    font-size: 0.75rem;
    opacity: 0.85;
  }

  /* Lightbox */

  .tcomic-lightbox {
    position: fixed;
    inset: 0;
    display: none;
    align-items: center;
    justify-content: center;
    z-index: 1000;
  }

  .tcomic-lightbox.is-open {
    display: flex;
  }

  .tcomic-lightbox-backdrop {
    position: absolute;
    inset: 0;
    background: rgba(0,0,0,0.8);
  }

  .tcomic-lightbox-inner {
    position: relative;
    max-width: 95vw;
    max-height: 90vh;
    padding: 0.75rem 0.75rem 0.9rem;
    border-radius: 18px;
    background: radial-gradient(circle at top left, rgba(168,85,247,0.24), rgba(15,23,42,0.98));
    box-shadow: 0 24px 48px rgba(0,0,0,0.85);
    border: 1px solid rgba(191,219,254,0.5);
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
    z-index: 1001;
  }

  .tcomic-lightbox-image {
    max-width: 100%;
    max-height: 70vh;
    border-radius: 12px;
    background: #020617;
    object-fit: contain;
  }

  .tcomic-lightbox-caption {
    font-size: 0.8rem;
    opacity: 0.9;
  }

  .tcomic-lightbox-caption::before {
    content: "🔍 ";
  }

  .tcomic-lightbox-download {
    align-self: flex-start;
  }

  .tcomic-lightbox-close {
    position: absolute;
    top: 0.25rem;
    right: 0.4rem;
    border: none;
    background: transparent;
    color: #e5e7eb;
    font-size: 1.4rem;
    line-height: 1;
    cursor: pointer;
    padding: 0.1rem 0.4rem;
  }

  .tcomic-lightbox-close:hover {
    filter: brightness(1.2);
  }
</style>

<script>
  (function () {
    const TOTAL_PAGES   = <?= (int)$totalPages; ?>;
    const BATCH_SIZE    = 20;
    const PAD_LENGTH    = <?= (int)$padLength; ?>;
    const FULL_FOLDER   = "<?= $fullFolderWeb; ?>";
    const THUMB_FOLDER  = "<?= $thumbFolderWeb; ?>";
    const FULL_EXT      = "<?= $fullExt; ?>";
    const THUMB_EXT     = "<?= $thumbExt; ?>";

    const gridEl   = document.getElementById("tcomicGrid");
    const statusEl = document.getElementById("tcomicStatus");
    const errorEl  = document.getElementById("tcomicError");
    const moreBtn  = document.getElementById("tcomicLoadMore");

    const lightbox          = document.getElementById("tcomicLightbox");
    const lightboxImg       = document.getElementById("tcomicLightboxImage");
    const lightboxCaption   = document.getElementById("tcomicLightboxCaption");
    const lightboxClose     = document.getElementById("tcomicLightboxClose");
    const lightboxDownload  = document.getElementById("tcomicLightboxDownload");

    if (!gridEl || !statusEl || !moreBtn) {
      return;
    }

    function pad(num, size) {
      let s = String(num);
      while (s.length < size) s = "0" + s;
      return s;
    }

    function fileBase(n) {
      return pad(n, PAD_LENGTH);
    }

    function thumbUrl(n) {
      return THUMB_FOLDER + fileBase(n) + THUMB_EXT;
    }

    function fullUrl(n) {
      return FULL_FOLDER + fileBase(n) + FULL_EXT;
    }

    const pages = [];
    for (let i = 1; i <= TOTAL_PAGES; i++) {
      const base = fileBase(i);
      pages.push({
        index: i,
        base: base,
        thumb: thumbUrl(i),
        full: fullUrl(i)
      });
    }
    pages.sort(function (a, b) { return a.base.localeCompare(b.base); });

    let rendered = 0;

    function updateStatus() {
      statusEl.textContent = "Showing " + rendered + " / " + TOTAL_PAGES + " pages 📖";
    }

    function openLightbox(page) {
      if (!lightbox || !lightboxImg || !lightboxCaption || !lightboxDownload) return;
      lightboxImg.src = page.full;
      lightboxImg.alt = "TComic page " + page.index;
      lightboxCaption.textContent = "Page " + page.index + " • " + page.base + FULL_EXT;
      lightboxDownload.href = page.full;
      lightbox.classList.add("is-open");
      lightbox.setAttribute("aria-hidden", "false");
    }

    function closeLightbox() {
      if (!lightbox || !lightboxImg) return;
      lightbox.classList.remove("is-open");
      lightbox.setAttribute("aria-hidden", "true");
      lightboxImg.src = "";
    }

    function renderNextBatch() {
      if (rendered >= pages.length) {
        moreBtn.disabled = true;
        moreBtn.textContent = "✅ All pages loaded";
        updateStatus();
        return;
      }

      const start = rendered;
      const end   = Math.min(rendered + BATCH_SIZE, pages.length);
      const slice = pages.slice(start, end);

      slice.forEach(function (page) {
        const card = document.createElement("a");
        card.href = page.full;
        card.className = "tcomic-card";

        const img = document.createElement("img");
        img.src = page.thumb;
        img.loading = "lazy";
        img.alt = "TComic page " + page.index;

        img.addEventListener("error", function () {
          if (!img.dataset.fallbackDone) {
            img.dataset.fallbackDone = "1";
            img.src = page.full; // fallback to full PNG if thumb missing
          } else {
            errorEl.textContent =
              "⚠️ Some images could not load. Check /tcomic and /tcomic/thumbs.";
          }
        });

        const meta = document.createElement("div");
        meta.className = "tcomic-meta";

        const left = document.createElement("span");
        left.className = "tcomic-meta-left";
        left.textContent = "Page " + page.index;

        const right = document.createElement("span");
        right.className = "tcomic-badge";
        right.textContent = page.base + FULL_EXT;

        meta.appendChild(left);
        meta.appendChild(right);

        card.appendChild(img);
        card.appendChild(meta);

        card.addEventListener("click", function (e) {
          e.preventDefault();
          openLightbox(page);
        });

        gridEl.appendChild(card);
      });

      rendered = end;
      updateStatus();
    }

    moreBtn.addEventListener("click", renderNextBatch);

    if (lightboxClose) {
      lightboxClose.addEventListener("click", function () {
        closeLightbox();
      });
    }

    if (lightbox) {
      lightbox.addEventListener("click", function (e) {
        if (e.target === lightbox || e.target.classList.contains("tcomic-lightbox-backdrop")) {
          closeLightbox();
        }
      });
    }

    document.addEventListener("keydown", function (e) {
      if (e.key === "Escape") {
        closeLightbox();
      }
    });

    // Initial batch
    updateStatus();
    renderNextBatch();
  })();
</script>
<?php
$console_body_html = ob_get_clean();
require __DIR__ . '/tshell.php';
