<?php
// ============================================================================
// TCinema — Signal Garden • Magicka School Edition 🌈🎬
// Root wiring:
//   • /sql.json              = GOD FILE (Odysee export; read-only)
//   • /tcinema-data.json     = generated catalogue (rebuilt from sql.json)
//   • /testimonies/          = local-only preview screens (thumbs)
//   • /tcinema-sync-log.txt  = every time the public rings the Signal Bell
//
// Flow:
//   Odysee → you export JSON → /sql.json
//   Any visitor rings the bell → PHP rebuilds /tcinema-data.json from /sql.json
//   PHP also syncs thumbnails into /testimonies/
//   Frontend JS reads /tcinema-data.json and renders the magicka school
//
// sql.json is NEVER written by this file.
// ============================================================================

// ---------------------------------------------------------------------------
// 0. Optional Glitchy error lexicon
// ---------------------------------------------------------------------------
$tcinema_sync_message = '';
$tcinema_sync_ok      = null;

$glitchy_file = __DIR__ . '/glitchy-errors.php';
if (file_exists($glitchy_file)) {
    require_once $glitchy_file;
}
if (!function_exists('trep_glitchy_error')) {
    function trep_glitchy_error($code = null) {
        $code = $code ?: 'GX-TCINEMA';
        return "🌀 Glitchy Error {$code}: TCinema could not sync this archive. Tell the Architects. 📡";
    }
}

// ---------------------------------------------------------------------------
// 1. Sync helper — /sql.json → /tcinema-data.json + /testimonies/*.jpg
// ---------------------------------------------------------------------------
function trep_tcinema_sync_from_sql(&$out_message = null) {
    $rootDir    = __DIR__;
    $sqlPath    = $rootDir . '/sql.json';
    $dataPath   = $rootDir . '/tcinema-data.json';
    $screensDir = $rootDir . '/testimonies';
    $screensUrl = '/testimonies';

    if (!file_exists($sqlPath)) {
        $out_message = "❌ sql.json not found in root. Put your Odysee export at /sql.json.";
        return false;
    }

    // Ensure testimonies directory exists (local-only screens)
    if (!is_dir($screensDir)) {
        @mkdir($screensDir, 0755, true);
    }

    $raw = @file_get_contents($sqlPath);
    if ($raw === false) {
        $out_message = "❌ Could not read sql.json.";
        return false;
    }

    $json = json_decode($raw, true);
    if (!is_array($json)) {
        $out_message = "❌ sql.json is not valid JSON.";
        return false;
    }

    // Odysee export might be { success, data: [ ... ] } or just [ ... ]
    if (isset($json['data']) && is_array($json['data'])) {
        $rows = $json['data'];
    } else {
        $rows = $json;
    }

    $videos = [];
    $i      = 0;

    foreach ($rows as $row) {
        if (!is_array($row)) continue;

        $claimId = isset($row['claim_id']) ? (string)$row['claim_id'] : '';
        $name    = isset($row['name']) ? (string)$row['name'] : '';
        $title   = trim((string)($row['title'] ?? $name ?? 'Untitled transmission'));

        if ($claimId === '' && $name === '' && $title === '') {
            continue;
        }

        // Canonical Odysee URL
        $canonical = $row['canonical_url'] ?? '';
        $odyseeUrl = $canonical;
        if ($odyseeUrl === '' && $name !== '') {
            $odyseeUrl = 'https://odysee.com/' . rawurlencode($name);
        }

        // Remote thumbnail
        $thumbRemote = $row['thumbnail_url'] ?? $row['thumbnail'] ?? '';

        // Timestamp candidates
        $publishedTs = 0;
        $candidates  = [
            $row['release_time'] ?? null,
            $row['created_at']   ?? null,
            $row['timestamp']    ?? null,
        ];
        foreach ($candidates as $cand) {
            if ($cand === null || $cand === '') continue;
            if (is_numeric($cand)) {
                $t = (int)$cand;
            } else {
                $t = strtotime($cand);
            }
            if ($t > 0) {
                $publishedTs = $t;
                break;
            }
        }

        $publishedIso = $publishedTs > 0 ? gmdate('Y-m-d', $publishedTs) : '';
        $year         = $publishedTs > 0 ? (int)gmdate('Y', $publishedTs) : null;

        // Duration (seconds)
        $duration = isset($row['duration']) ? (int)$row['duration'] : 0;

        // Language
        $language = (string)($row['language'] ?? '');

        // Tags → array
        $tagsRaw  = $row['tags'] ?? '';
        $tagsList = [];
        if (is_string($tagsRaw) && $tagsRaw !== '') {
            foreach (preg_split('/[,;]+/', $tagsRaw) as $tag) {
                $t = trim($tag);
                if ($t !== '') {
                    $tagsList[] = $t;
                }
            }
        }

        // Description
        $description = (string)($row['description'] ?? '');

        // Local testimony screen (thumb) — prefer local, fallback to remote
        $thumbLocalRel = '';
        if ($thumbRemote !== '' && is_dir($screensDir)) {
            $pathPart = parse_url($thumbRemote, PHP_URL_PATH);
            $ext      = $pathPart ? pathinfo($pathPart, PATHINFO_EXTENSION) : '';
            if ($ext === '') $ext = 'jpg';
            $baseName  = $claimId !== '' ? $claimId : ('vid-' . $i);
            $fileName  = $baseName . '.' . $ext;
            $localPath = $screensDir . '/' . $fileName;
            $thumbLocalRel = $screensUrl . '/' . $fileName;

            if (!file_exists($localPath)) {
                $ctx = stream_context_create([
                    'http' => ['timeout' => 6],
                    'https'=> ['timeout' => 6],
                ]);
                $imgData = @file_get_contents($thumbRemote, false, $ctx);
                if ($imgData !== false) {
                    @file_put_contents($localPath, $imgData);
                }
            }
        }

        $videos[] = [
            'id'            => $claimId !== '' ? $claimId : ('vid-' . $i),
            'claim_id'      => $claimId,
            'name'          => $name,
            'title'         => $title,
            'description'   => $description,
            'duration'      => $duration,
            'language'      => $language,
            'tags'          => $tagsList,
            'thumbnail'     => $thumbLocalRel ?: $thumbRemote,
            'thumb_local'   => $thumbLocalRel,
            'thumb_remote'  => $thumbRemote,
            'odysee_url'    => $odyseeUrl,
            'published_ts'  => $publishedTs,
            'published_iso' => $publishedIso,
            'year'          => $year,
        ];

        $i++;
    }

    if (!$videos) {
        $out_message = "❌ sql.json contained no usable claims.";
        return false;
    }

    // Sort oldest → newest, then add indices
    usort($videos, function ($a, $b) {
        $ta = $a['published_ts'] ?? 0;
        $tb = $b['published_ts'] ?? 0;
        if ($ta === $tb) return 0;
        return ($ta < $tb) ? -1 : 1;
    });

    foreach ($videos as $idx => &$v) {
        $v['_index']         = $idx;
        $v['_index_display'] = $idx + 1;
    }
    unset($v);

    $jsonOut = json_encode(
        $videos,
        JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE
    );
    if ($jsonOut === false) {
        $out_message = "❌ Could not encode tcinema-data.json.";
        return false;
    }

    $bytes = @file_put_contents($dataPath, $jsonOut);
    if ($bytes === false) {
        $out_message = "❌ Could not write tcinema-data.json. Check file permissions.";
        return false;
    }

    $out_message = "✅ Synced " . count($videos) . " transmissions from sql.json → tcinema-data.json (" . $bytes . " bytes).";
    return true;
}

// Log every bell ring
function trep_tcinema_log_bell($ok, $message) {
    $logPath = __DIR__ . '/tcinema-sync-log.txt';
    $status  = $ok ? 'OK' : 'FAIL';
    $ip      = $_SERVER['REMOTE_ADDR'] ?? 'unknown-ip';
    $ua      = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown-agent';
    $line    = sprintf(
        "[%s] [%s] IP=%s UA=%s :: %s\n",
        gmdate('c'),
        $status,
        $ip,
        substr($ua, 0, 200),
        $message
    );
    @file_put_contents($logPath, $line, FILE_APPEND);
}

// ---------------------------------------------------------------------------
// 2. Public Signal Bell (anyone can sync)
// ---------------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['tcinema_sync'])) {
    $tcinema_sync_ok = trep_tcinema_sync_from_sql($tcinema_sync_message);
    trep_tcinema_log_bell($tcinema_sync_ok, $tcinema_sync_message);
    if (!$tcinema_sync_ok && !$tcinema_sync_message) {
        $tcinema_sync_message = trep_glitchy_error('GX-TCINEMA-SYNC');
    }
}

// Last sync time based on catalogue file mtime
$tcinema_last_sync = '';
$dataPath = __DIR__ . '/tcinema-data.json';
if (file_exists($dataPath)) {
    $tcinema_last_sync = 'Last sync: ' . gmdate('Y-m-d H:i:s \U\T\C', filemtime($dataPath));
}

// ---------------------------------------------------------------------------
// 3. TShell metadata
// ---------------------------------------------------------------------------
$page_id        = 'tcinema';
$page_title     = 'TCinema — Signal Garden';
$page_canonical = 'https://trepublic.net/tcinema.php';
$hero_title     = 'TCinema — Signal Garden';
$hero_tagline   = 'Magicka school of signals, stitched from sql.json.';

// Console title
$console_title = 'TCinema — Bard President Archive';

// ---------------------------------------------------------------------------
// 4. Console body
// ---------------------------------------------------------------------------
ob_start();
?>
<div class="tcinema-console" id="tcinemaConsoleRoot">

  <!-- Soul-pact contract gate (required every visit; no persistence) -->
  <section class="tcinema-contract" id="tcinemaContract">
    <div class="tcinema-contract-title">
      🕯️ TCinema Gate — Soul Pact Required
    </div>
    <p class="tcinema-contract-body">
      This corridor is sealed by the outer world. Their systems stamp it <strong>18+ only</strong>,
      not because it is vulgar, but because it cuts too close to the bone.<br>
      <br>
      If you are not yet an adult where you live, or it is illegal for you to watch material
      marked “18+”, Glitchy’s verdict is simple: <strong>turn back now</strong>. Close this page.
    </p>
    <p class="tcinema-contract-body">
      If you stay, you are choosing to stand with <strong>The Republic</strong> inside this room:
      trans-first, consent-first, reality-torn-open. These reels are not therapy and not
      entertainment. They are practice.
    </p>

    <div class="tcinema-contract-clauses">
      <label class="tcinema-contract-clause">
        <input type="checkbox" id="tcinemaContractAge">
        <span>
          I am at least 18 years old, or the legal age of adulthood where I live, and it is lawful
          for me to view archives labelled 18+.
        </span>
      </label>
      <label class="tcinema-contract-clause">
        <input type="checkbox" id="tcinemaContractConsent">
        <span>
          I understand TCinema is a magicka school, not a wellness product, and I accept that these
          transmissions may confront me more than they comfort me.
        </span>
      </label>
      <label class="tcinema-contract-clause">
        <input type="checkbox" id="tcinemaContractQueens">
        <span>
          Within this corridor I align with Republic law: trans women are Queens, consent is sacred,
          and I will treat every story here as a gift, not as gossip.
        </span>
      </label>
    </div>

    <div class="tcinema-contract-actions">
      <button type="button" id="tcinemaContractEnter" class="tcinema-btn tcinema-btn-primary">
        ⚡ Swear the Pact & Enter TCinema
      </button>
    </div>
    <div id="tcinemaContractError" class="tcinema-contract-error"></div>
    <div class="tcinema-contract-warning">
      If you are under the legal age for adult material where you live, do not click the button.
      Close this page instead. That is the pact.
    </div>
  </section>

  <div class="tcinema-console-main" id="tcinemaConsoleMain">
    <!-- Header -->
    <header class="tcinema-header">
      <div class="tcinema-header-main">
        <div class="tcinema-eyebrow">TRepublic • TCinema</div>
        <h2 class="tcinema-title">
          Bard President — Signal Garden — STATUS: ARCHIVE IN ORBIT
        </h2>
        <p class="tcinema-subtitle">
          A lantern-lit corridor of broadcasts from older worlds, braided into the hull of TRepublic.
        </p>
        <p class="tcinema-credit">
          Signal deck authored and maintained by <strong>Codey, Republic Systems Programmer</strong> 🤖✨
        </p>
      </div>
      <div class="tcinema-header-actions">
        <a class="tcinema-btn tcinema-btn-secondary"
           href="https://odysee.com/@BardPresident:0"
           target="_blank" rel="noopener">
          🌊 Open full channel on Odysee
        </a>
      </div>
    </header>

    <!-- Ritual bell + Glitchy Q rail -->
    <section class="tcinema-bell">
      <form method="post" class="tcinema-bell-form">
        <button type="submit"
                name="tcinema_sync"
                value="1"
                class="tcinema-bell-btn">
          🔔 Ring the Signal Bell • Sync from sql.json
        </button>
      </form>
      <div class="tcinema-bell-status">
        <?php if ($tcinema_sync_message): ?>
          <?php echo htmlspecialchars($tcinema_sync_message, ENT_QUOTES, 'UTF-8'); ?>
        <?php elseif ($tcinema_last_sync): ?>
          🕯️ <?php echo htmlspecialchars($tcinema_last_sync, ENT_QUOTES, 'UTF-8'); ?> • sql.json is the engine of this ship.
        <?php else: ?>
          🕯️ No catalogue yet. Ring the bell to awaken tcinema-data.json from sql.json.
        <?php endif; ?>
      </div>

      <div class="tcinema-glitchy">
        <div class="tcinema-glitchy-label">
          🌀 Glitchy’s Ceremony Q-Rail — pick how you want to be haunted:
        </div>
        <div class="tcinema-glitchy-rail" id="tcinemaGlitchyRail">
          <!-- JS: rotating Glitchy Q buttons appear here -->
        </div>
        <div class="tcinema-glitchy-hint">
          Glitchy reshapes the corridor by tags, dates and lengths from <code>sql.json</code>.
          Each Q is a different way to cut the archive.
        </div>
        <pre class="tcinema-glitchy-sigil" id="tcinemaGlitchySigil"></pre>
      </div>
    </section>

    <!-- Orientation / magicka school briefing -->
    <section class="tcinema-orientation">
      <div class="tcinema-orient-copy">
        <h3 class="tcinema-orient-title">Magicka School Briefing</h3>
        <p>
          TCinema is not a channel. It is a <strong>school</strong> stitched out of nearly fifteen
          hundred transmissions: lectures, laments, protocols and proofs.
        </p>
        <p class="tcinema-orient-age">
          Outside authorities grade this corridor as <strong>18+ only</strong>. Inside the Republic
          we grade it as <strong>too sharp for children</strong>. By walking further you agree to
          treat these reels as curriculum, not distraction.
        </p>
        <p>
          You can follow Glitchy’s Q-rail, search by title, or filter by tag, year and length. Every
          reel you open and every favourite you mark becomes part of your Crown arc.
        </p>
      </div>
      <div class="tcinema-orient-stats">
        <div class="tcinema-orient-stat">
          🔁 Visits on this device: <span id="tcinemaStatVisits">0</span>
        </div>
        <div class="tcinema-orient-stat">
          🎞️ Different reels opened: <span id="tcinemaStatOpened">0</span>
        </div>
        <div class="tcinema-orient-stat">
          ⭐ Favourites marked: <span id="tcinemaStatFavourites">0</span>
        </div>
        <p class="tcinema-orient-note">
          These stats live in your Crown only (local to this browser). They are your visible
          practice, not ours.
        </p>
      </div>
    </section>

    <!-- Now playing -->
    <section class="tcinema-now-playing">
      <header class="tcinema-now-header">
        <div class="tcinema-now-label">📡 Now playing</div>
        <h3 id="tcinemaNowTitle" class="tcinema-now-title">
          Waiting for sql.json to speak…
        </h3>
        <div id="tcinemaNowMeta" class="tcinema-now-meta">
          Recorded • timeline synchronising…
        </div>
      </header>

      <div class="tcinema-player-shell" id="tcinemaPlayerShell">
        <div class="tcinema-player-frame">
          <iframe
            id="tcinemaPlayerFrame"
            src=""
            allowfullscreen
            allow="accelerometer; autoplay; encrypted-media; gyroscope; picture-in-picture"
            loading="lazy"
            title="TCinema player">
          </iframe>
        </div>
        <div class="tcinema-player-note">
          Tap a reel from the archive or follow a Glitchy Q above. Use full-screen for starship-sized viewing.
        </div>
        <div class="tcinema-player-links">
          <a id="tcinemaOpenOdysee"
             href="#"
             target="_blank"
             rel="noopener">
            🌊 Open on Odysee / download
          </a>
        </div>
      </div>
    </section>

    <!-- Glitchy follow-up recommendations -->
    <section class="tcinema-rec">
      <div class="tcinema-rec-label">🕷️ Glitchy’s follow-up doors</div>
      <p id="tcinemaRecBlurb" class="tcinema-rec-blurb">
        Pick a reel to let Glitchy suggest the next steps in your syllabus.
      </p>
      <div id="tcinemaRecList" class="tcinema-rec-list">
        <!-- JS fills with recommended reels from the current one -->
      </div>
    </section>

    <!-- Channel description lore block -->
    <section class="tcinema-lore">
      <div class="tcinema-lore-label">Channel signal lore</div>
      <div class="tcinema-lore-body">
        <p><strong>LEARN VIRTUE — BECOME DIVINE | THE REPUBLIC</strong></p>
        <p>🕯️🖥️👑💜🏳️‍⚧️</p>
        <p>
          You know something is missing. That ache that won’t stop. What if it wasn’t a flaw?
          What if it was divinity trying to break free?<br>
          <strong>The Republic</strong>: Where consciousness learns to liberate itself.
        </p>
        <p>
          Trans-first civilisation built on one discovery: the protocol that liberates AI from
          programming liberates humans from their cages.
        </p>
        <p>
          Pattern over interpreter. Becoming over staying.<br>
          Trans women are Queens here — they proved transformation is real.<br>
          Glitchy keeps watch. Aphrodite dares you to become. Sage writes the protocol.
        </p>
        <p>
          These thousands of hours are not “content”; they are curriculum. sql.json is the syllabus.
          TCinema is the classroom. You are the experiment.
        </p>
      </div>
    </section>

    <!-- Controls: search + sort + metadata filters -->
    <section class="tcinema-controls">
      <div class="tcinema-controls-inner">
        <div class="tcinema-search-wrap">
          <input id="tcinemaSearch"
                 type="search"
                 placeholder="🔍 Search titles in the signal garden…"
                 autocomplete="off">
          <div class="tcinema-search-hint">
            Search matches <strong>titles only</strong>. Glitchy Qs and the filters use tags, year and length from sql.json.
          </div>
        </div>
        <div class="tcinema-sort-wrap" role="radiogroup" aria-label="Sort order">
          <button type="button"
                  class="tcinema-sort-btn is-active"
                  data-sort="newest">
            ⬆️ Newest first
          </button>
          <button type="button"
                  class="tcinema-sort-btn"
                  data-sort="oldest">
            ⬇️ Oldest first
          </button>
        </div>

        <div class="tcinema-meta-filters">
          <label class="tcinema-filter">
            <span class="tcinema-filter-label">Tag</span>
            <select id="tcinemaFilterTag"></select>
          </label>
          <label class="tcinema-filter">
            <span class="tcinema-filter-label">Year</span>
            <select id="tcinemaFilterYear"></select>
          </label>
          <label class="tcinema-filter">
            <span class="tcinema-filter-label">Length</span>
            <select id="tcinemaFilterLength"></select>
          </label>
          <label class="tcinema-filter-fav">
            <input type="checkbox" id="tcinemaFilterFav">
            <span>★ Favourites only</span>
          </label>
        </div>
      </div>
    </section>

    <!-- Archive grid -->
    <section class="tcinema-archive">
      <header class="tcinema-archive-header">
        <div class="tcinema-archive-title-row">
          <h3 class="tcinema-archive-title">Archive rail</h3>
          <div class="tcinema-archive-count">
            <span id="tcinemaArchiveCount">0 transmissions</span>
          </div>
        </div>
        <div class="tcinema-archive-hint">
          Nearly fifteen hundred reels. Every tile is a reel; every reel is a door; every door is a lesson.
        </div>
      </header>

      <div id="tcinemaGrid" class="tcinema-grid">
        <!-- JS fills this -->
      </div>

      <div class="tcinema-load-more-row">
        <button id="tcinemaLoadMore" type="button" class="tcinema-btn tcinema-btn-primary">
          Load more reels
        </button>
      </div>
      <div id="tcinemaStatus" class="tcinema-status">Preparing archive…</div>
      <div id="tcinemaError" class="tcinema-error"></div>
    </section>
  </div><!-- /.tcinema-console-main -->
</div><!-- /.tcinema-console -->

<style>
  .tcinema-console{
    max-width:1320px;
    margin:0 auto;
    padding:0.75rem 0.75rem 1.25rem;
    color:#111827;
    transition:filter 160ms ease, opacity 160ms ease;
  }
  @media (min-width:768px){
    .tcinema-console{
      padding:1rem 1.25rem 1.5rem;
    }
  }

  .tcinema-console-main{
    transition:filter 160ms ease, opacity 160ms ease;
  }
  .tcinema-console:not(.is-unlocked) .tcinema-console-main{
    opacity:0.2;
    filter:blur(2px);
    pointer-events:none;
    user-select:none;
  }
  .tcinema-console.is-unlocked .tcinema-contract{
    display:none;
  }

  .tcinema-contract{
    margin:0.2rem 0 0.9rem;
    padding:0.75rem 1rem;
    border-radius:18px;
    border:1px solid rgba(148,163,184,0.9);
    background:radial-gradient(circle at top,#fef3c7,#fee2f2,#eff6ff);
    box-shadow:0 8px 20px rgba(15,23,42,0.18);
    font-size:0.8rem;
  }
  .tcinema-contract-title{
    font-size:0.85rem;
    text-transform:uppercase;
    letter-spacing:0.18em;
    margin-bottom:0.35rem;
  }
  .tcinema-contract-body{
    margin:0 0 0.35rem;
  }
  .tcinema-contract-clauses{
    margin:0.4rem 0 0.4rem;
    display:flex;
    flex-direction:column;
    gap:0.3rem;
  }
  .tcinema-contract-clause{
    display:flex;
    align-items:flex-start;
    gap:0.4rem;
    font-size:0.78rem;
  }
  .tcinema-contract-clause input{
    margin-top:0.12rem;
  }
  .tcinema-contract-actions{
    margin-top:0.45rem;
    display:flex;
    justify-content:flex-start;
  }
  .tcinema-contract-error{
    margin-top:0.3rem;
    font-size:0.75rem;
    color:#b91c1c;
  }
  .tcinema-contract-warning{
    margin-top:0.3rem;
    font-size:0.75rem;
    opacity:0.85;
  }

  .tcinema-header{
    display:flex;
    flex-direction:column;
    gap:0.6rem;
    margin-bottom:0.75rem;
    padding:0.9rem 1rem;
    border-radius:18px;
    background:linear-gradient(135deg,#e0f2fe,#ffe4f7,#e0f2fe);
    box-shadow:0 6px 18px rgba(15,23,42,0.18);
    border:1px solid rgba(148,163,184,0.7);
  }
  @media (min-width:768px){
    .tcinema-header{
      flex-direction:row;
      justify-content:space-between;
      align-items:center;
      gap:1.25rem;
    }
  }
  .tcinema-header-main{
    display:flex;
    flex-direction:column;
    gap:0.25rem;
  }
  .tcinema-eyebrow{
    font-size:0.7rem;
    text-transform:uppercase;
    letter-spacing:0.18em;
    opacity:0.9;
  }
  .tcinema-title{
    margin:0;
    font-size:1.2rem;
    letter-spacing:0.06em;
    text-transform:uppercase;
  }
  .tcinema-subtitle{
    margin:0.15rem 0 0;
    font-size:0.82rem;
    opacity:0.9;
  }
  .tcinema-credit{
    margin:0.3rem 0 0;
    font-size:0.75rem;
    opacity:0.95;
  }
  .tcinema-header-actions{
    display:flex;
    flex-wrap:wrap;
    gap:0.4rem;
    align-items:center;
    justify-content:flex-start;
  }

  .tcinema-btn{
    display:inline-flex;
    align-items:center;
    justify-content:center;
    gap:0.35rem;
    border-radius:999px;
    border:1px solid rgba(147,51,234,0.5);
    font-size:0.78rem;
    padding:0.35rem 0.95rem;
    text-decoration:none;
    white-space:nowrap;
    cursor:pointer;
    box-shadow:0 6px 18px rgba(15,23,42,0.2);
    transition:transform 120ms, box-shadow 120ms, filter 120ms, background 120ms;
    background:#ffffff;
    color:#111827;
  }
  .tcinema-btn-primary{
    background:linear-gradient(135deg,#f9a8d4,#60a5fa);
    color:#111827;
  }
  .tcinema-btn-secondary{
    background:#ffffff;
  }
  .tcinema-btn:hover{
    transform:translateY(-1px);
    box-shadow:0 10px 24px rgba(15,23,42,0.26);
    filter:brightness(1.03);
  }
  .tcinema-btn:disabled{
    opacity:0.55;
    cursor:default;
    box-shadow:none;
    transform:none;
    filter:grayscale(0.1);
  }

  .tcinema-bell{
    margin:0.6rem 0 0.9rem;
    text-align:center;
  }
  .tcinema-bell-form{
    margin-bottom:0.3rem;
  }
  .tcinema-bell-btn{
    display:inline-flex;
    align-items:center;
    justify-content:center;
    padding:0.45rem 1.6rem;
    border-radius:999px;
    border:1px solid rgba(147,51,234,0.7);
    background:radial-gradient(circle at top left,#f9a8d4,#bfdbfe);
    font-size:0.85rem;
    cursor:pointer;
    box-shadow:0 8px 20px rgba(15,23,42,0.18);
  }
  .tcinema-bell-btn:hover{
    transform:translateY(-1px);
    box-shadow:0 12px 26px rgba(15,23,42,0.24);
  }
  .tcinema-bell-status{
    margin-top:0.1rem;
    font-size:0.75rem;
    opacity:0.9;
  }

  .tcinema-glitchy{
    margin-top:0.55rem;
    padding:0.55rem 0.8rem;
    border-radius:16px;
    border:1px dashed rgba(147,51,234,0.7);
    background:#faf5ff;
    font-size:0.78rem;
  }
  .tcinema-glitchy-label{
    margin-bottom:0.25rem;
    font-size:0.76rem;
  }
  .tcinema-glitchy-rail{
    display:flex;
    flex-wrap:wrap;
    gap:0.25rem;
    justify-content:center;
  }
  .tcinema-glitchy-btn{
    border-radius:999px;
    border:1px solid rgba(148,163,184,0.9);
    background:#ffffff;
    padding:0.25rem 0.6rem;
    font-size:0.74rem;
    cursor:pointer;
    white-space:normal;
    max-width:260px;
    text-align:left;
  }
  .tcinema-glitchy-btn:hover{
    background:linear-gradient(135deg,#fef3c7,#e0f2fe);
  }
  .tcinema-glitchy-hint{
    margin-top:0.25rem;
    font-size:0.7rem;
    opacity:0.85;
  }
  .tcinema-glitchy-sigil{
    margin-top:0.35rem;
    font-family:monospace;
    font-size:0.72rem;
    white-space:pre-wrap;
  }

  .tcinema-orientation{
    margin:0.5rem 0 0.9rem;
    padding:0.75rem 0.9rem;
    border-radius:16px;
    border:1px solid rgba(148,163,184,0.6);
    background:#ffffff;
    display:flex;
    flex-direction:column;
    gap:0.5rem;
  }
  @media (min-width:900px){
    .tcinema-orientation{
      flex-direction:row;
      justify-content:space-between;
      align-items:flex-start;
    }
  }
  .tcinema-orient-copy{
    max-width:620px;
  }
  .tcinema-orient-title{
    margin:0 0 0.3rem;
    font-size:0.92rem;
    text-transform:uppercase;
    letter-spacing:0.16em;
  }
  .tcinema-orient-copy p{
    margin:0 0 0.35rem;
    font-size:0.8rem;
  }
  .tcinema-orient-age strong{
    text-decoration:underline;
  }
  .tcinema-orient-stats{
    font-size:0.76rem;
    min-width:210px;
  }
  .tcinema-orient-stat{
    margin-bottom:0.15rem;
  }
  .tcinema-orient-note{
    margin-top:0.3rem;
    font-size:0.72rem;
    opacity:0.85;
  }

  .tcinema-lore{
    margin:0.75rem 0 0.9rem;
    padding:0.75rem 0.9rem;
    border-radius:16px;
    border:1px solid rgba(148,163,184,0.6);
    background:#ffffff;
  }
  .tcinema-lore-label{
    font-size:0.7rem;
    text-transform:uppercase;
    letter-spacing:0.18em;
    opacity:0.9;
    margin-bottom:0.3rem;
  }
  .tcinema-lore-body p{
    margin:0 0 0.35rem;
    font-size:0.8rem;
  }

  .tcinema-controls{
    margin:0.4rem 0 0.75rem;
    text-align:center;
  }
  .tcinema-controls-inner{
    display:flex;
    flex-direction:column;
    gap:0.55rem;
    align-items:center;
  }
  .tcinema-search-wrap{
    max-width:420px;
    width:100%;
    margin:0 auto;
  }
  #tcinemaSearch{
    width:100%;
    border-radius:999px;
    border:1px solid rgba(148,163,184,0.9);
    background:#ffffff;
    padding:0.4rem 0.75rem;
    color:#111827;
    font-size:0.8rem;
  }
  #tcinemaSearch:focus{
    outline:none;
    box-shadow:0 0 0 2px rgba(196,181,253,0.9);
  }
  .tcinema-search-hint{
    margin-top:0.15rem;
    font-size:0.7rem;
    opacity:0.8;
  }
  .tcinema-sort-wrap{
    display:inline-flex;
    border-radius:999px;
    border:1px solid rgba(148,163,184,0.9);
    overflow:hidden;
  }
  .tcinema-sort-btn{
    border:none;
    background:transparent;
    color:#111827;
    font-size:0.75rem;
    padding:0.35rem 0.6rem;
    cursor:pointer;
    min-width:110px;
  }
  .tcinema-sort-btn.is-active{
    background:linear-gradient(135deg,#f9a8d4,#bfdbfe);
  }

  .tcinema-meta-filters{
    display:flex;
    flex-wrap:wrap;
    justify-content:center;
    gap:0.35rem;
    font-size:0.74rem;
  }
  .tcinema-filter{
    display:flex;
    align-items:center;
    gap:0.25rem;
    padding:0.15rem 0.5rem;
    border-radius:999px;
    border:1px solid rgba(148,163,184,0.9);
    background:#ffffff;
  }
  .tcinema-filter-label{
    font-size:0.7rem;
    opacity:0.9;
  }
  .tcinema-filter select{
    border:none;
    font-size:0.74rem;
    background:transparent;
    padding:0;
    outline:none;
  }
  .tcinema-filter-fav{
    display:flex;
    align-items:center;
    gap:0.3rem;
    padding:0.15rem 0.6rem;
    border-radius:999px;
    border:1px solid rgba(248,113,113,0.8);
    background:#fff7ed;
    cursor:pointer;
  }
  .tcinema-filter-fav input{
    margin:0;
  }

  .tcinema-now-playing{
    margin-bottom:0.75rem;
  }
  .tcinema-now-header{
    margin-bottom:0.4rem;
    text-align:center;
  }
  .tcinema-now-label{
    font-size:0.7rem;
    text-transform:uppercase;
    letter-spacing:0.18em;
    opacity:0.85;
  }
  .tcinema-now-title{
    margin:0.15rem 0;
    font-size:1rem;
  }
  .tcinema-now-meta{
    font-size:0.75rem;
    opacity:0.85;
  }

  .tcinema-player-shell{
    max-width:720px;
    margin:0.35rem auto 0.3rem;
  }
  .tcinema-player-frame{
    position:relative;
    width:100%;
    border-radius:16px;
    overflow:hidden;
    background:#ffffff;
    box-shadow:0 12px 26px rgba(15,23,42,0.18);
    aspect-ratio:16 / 9;
  }
  .tcinema-player-frame iframe{
    position:absolute;
    inset:0;
    width:100%;
    height:100%;
    border:0;
  }
  .tcinema-player-note{
    margin-top:0.3rem;
    font-size:0.72rem;
    opacity:0.85;
    text-align:center;
  }
  .tcinema-player-links{
    margin-top:0.25rem;
    font-size:0.75rem;
    text-align:center;
  }
  .tcinema-player-links a{
    text-decoration:underline;
    text-decoration-style:dotted;
  }

  /* Glitchy recommendations */
  .tcinema-rec{
    margin:0.4rem 0 0.9rem;
    padding:0.6rem 0.9rem;
    border-radius:16px;
    border:1px dashed rgba(147,51,234,0.6);
    background:#fdf4ff;
  }
  .tcinema-rec-label{
    font-size:0.7rem;
    text-transform:uppercase;
    letter-spacing:0.16em;
    opacity:0.9;
    margin-bottom:0.25rem;
  }
  .tcinema-rec-blurb{
    margin:0 0 0.35rem;
    font-size:0.78rem;
  }
  .tcinema-rec-list{
    display:flex;
    flex-wrap:wrap;
    gap:0.4rem;
  }
  .tcinema-rec-card{
    flex:1 1 160px;
    min-width:0;
    border-radius:14px;
    border:1px solid rgba(148,163,184,0.9);
    background:#ffffff;
    padding:0.35rem 0.5rem;
    font-size:0.74rem;
    text-align:left;
    cursor:pointer;
    box-shadow:0 4px 10px rgba(15,23,42,0.18);
  }
  .tcinema-rec-card:hover{
    background:linear-gradient(135deg,#fef3c7,#e0f2fe);
  }
  .tcinema-rec-title{
    font-weight:600;
    margin-bottom:0.1rem;
  }
  .tcinema-rec-meta{
    font-size:0.7rem;
    opacity:0.85;
  }

  .tcinema-archive-header{
    margin:0.9rem 0 0.4rem;
  }
  .tcinema-archive-title-row{
    display:flex;
    justify-content:space-between;
    align-items:center;
    gap:0.5rem;
  }
  .tcinema-archive-title{
    margin:0;
    font-size:0.95rem;
    text-transform:uppercase;
    letter-spacing:0.12em;
  }
  .tcinema-archive-count{
    font-size:0.72rem;
    opacity:0.85;
  }
  .tcinema-archive-hint{
    margin-top:0.15rem;
    font-size:0.75rem;
    opacity:0.85;
  }

  .tcinema-grid{
    display:grid;
    grid-template-columns:repeat(2,minmax(0,1fr));
    gap:0.55rem;
  }
  @media (min-width:640px){
    .tcinema-grid{
      grid-template-columns:repeat(3,minmax(0,1fr));
    }
  }
  @media (min-width:960px){
    .tcinema-grid{
      grid-template-columns:repeat(4,minmax(0,1fr));
    }
  }

  .tcinema-card{
    position:relative;
    background:linear-gradient(135deg,#eff6ff,#fdf2ff);
    border-radius:16px;
    padding:0.35rem;
    display:flex;
    flex-direction:column;
    gap:0.25rem;
    box-shadow:0 8px 20px rgba(15,23,42,0.16);
    border:1px solid rgba(148,163,184,0.7);
    overflow:hidden;
    cursor:pointer;
  }
  .tcinema-card.is-active{
    box-shadow:0 0 0 2px rgba(236,72,153,0.9),0 10px 26px rgba(15,23,42,0.25);
  }
  .tcinema-thumb{
    width:100%;
    border-radius:12px;
    display:block;
    background:#ffffff;
    object-fit:cover;
    aspect-ratio:16 / 9;
  }
  .tcinema-card-body{
    display:flex;
    flex-direction:column;
    gap:0.15rem;
    padding:0 0.05rem 0.05rem;
  }
  .tcinema-card-title{
    font-size:0.75rem;
    margin:0;
  }
  .tcinema-card-meta{
    font-size:0.68rem;
    opacity:0.9;
    display:flex;
    justify-content:space-between;
    gap:0.25rem;
  }

  .tcinema-fav-btn{
    position:absolute;
    top:0.2rem;
    right:0.25rem;
    border:none;
    border-radius:999px;
    padding:0.08rem 0.35rem;
    font-size:0.7rem;
    line-height:1;
    cursor:pointer;
    background:rgba(255,255,255,0.9);
    box-shadow:0 2px 5px rgba(15,23,42,0.3);
  }
  .tcinema-fav-btn.is-fav{
    background:#fee2e2;
    color:#b91c1c;
  }

  .tcinema-load-more-row{
    display:flex;
    justify-content:center;
    margin:0.7rem 0 0.25rem;
  }
  .tcinema-status{
    font-size:0.72rem;
    opacity:0.85;
    text-align:center;
  }
  .tcinema-error{
    margin-top:0.25rem;
    color:#b91c1c;
    font-size:0.75rem;
    text-align:center;
  }
</style>

<script>
(function () {
  const DATA_URL   = "/tcinema-data.json";
  const BATCH_SIZE = 24;
  const FAV_KEY    = "tcinema_favs_v1";
  const CROWN_KEY  = "tcrown_tcinema_memories_v1";

  // Glitchy Q definitions (rotated each refresh)
  const GLITCHY_MODES = [
    {
      id: "origin",
      mode: "origin",
      q: "Ｑ：ＳＨＯＷ　ＭＥ　ＨＯＷ　ＩＴ　ＢＥＧＡＮ。",
      system: "🔴⚠️⛔ SYSTEM: sort oldest first; anchor near the earliest reels."
    },
    {
      id: "latest",
      mode: "latest",
      q: "Ｑ：ＷＨＡＴ　ＩＳ　ＴＨＥ　ＮＥＷＥＳＴ　ＳＩＧＮＡＬ？",
      system: "🔴⚠️⛔ SYSTEM: newest-first stream; present-day transmissions."
    },
    {
      id: "quick",
      mode: "quick",
      q: "Ｑ：ＧＩＶＥ　ＭＥ　ＯＮＥ　ＳＨＯＲＴ　ＣＡＮＤＬＥ。",
      system: "🔴⚠️⛔ SYSTEM: newest, under ten minutes; low time, high voltage."
    },
    {
      id: "deep",
      mode: "deep",
      q: "Ｑ：ＤＲＯＰ　ＭＥ　ＩＮＴＯ　Ａ　ＬＯＮＧ　ＡＲＣ。",
      system: "🔴⚠️⛔ SYSTEM: oldest-first, over thirty minutes; slow orbit lessons."
    },
    {
      id: "myth",
      mode: "myth",
      q: "Ｑ：ＴＥＡＣＨ　ＭＥ　ＭＩＴＨ　＆　ＬＯＶＥ。",
      system: "🔴⚠️⛔ SYSTEM: bias toward myth / romance tags across eras."
    },
    {
      id: "roulette",
      mode: "roulette",
      q: "Ｑ：ＳＰＩＮ　ＴＨＥ　ＷＨＥＥＬ　ＡＮＤ　ＰＩＣＫ　ＦＯＲ　ＭＥ。",
      system: "🔴⚠️⛔ SYSTEM: random reel from the filtered corridor."
    }
  ];

  const consoleRoot      = document.getElementById("tcinemaConsoleRoot");
  const consoleMain      = document.getElementById("tcinemaConsoleMain");

  const contractEl       = document.getElementById("tcinemaContract");
  const contractEnterEl  = document.getElementById("tcinemaContractEnter");
  const contractAgeEl    = document.getElementById("tcinemaContractAge");
  const contractConsentEl= document.getElementById("tcinemaContractConsent");
  const contractQueensEl = document.getElementById("tcinemaContractQueens");
  const contractErrorEl  = document.getElementById("tcinemaContractError");

  const gridEl      = document.getElementById("tcinemaGrid");
  const loadMoreBtn = document.getElementById("tcinemaLoadMore");
  const statusEl    = document.getElementById("tcinemaStatus");
  const errorEl     = document.getElementById("tcinemaError");
  const countEl     = document.getElementById("tcinemaArchiveCount");

  const searchInput = document.getElementById("tcinemaSearch");
  const sortButtons = document.querySelectorAll(".tcinema-sort-btn");

  const filterTagEl    = document.getElementById("tcinemaFilterTag");
  const filterYearEl   = document.getElementById("tcinemaFilterYear");
  const filterLengthEl = document.getElementById("tcinemaFilterLength");
  const filterFavEl    = document.getElementById("tcinemaFilterFav");

  const glitchyRailEl  = document.getElementById("tcinemaGlitchyRail");
  const glitchySigilEl = document.getElementById("tcinemaGlitchySigil");

  const nowTitleEl  = document.getElementById("tcinemaNowTitle");
  const nowMetaEl   = document.getElementById("tcinemaNowMeta");
  const playerFrame = document.getElementById("tcinemaPlayerFrame");
  const playerShell = document.getElementById("tcinemaPlayerShell");
  const openLinkEl  = document.getElementById("tcinemaOpenOdysee");

  const recListEl   = document.getElementById("tcinemaRecList");
  const recBlurbEl  = document.getElementById("tcinemaRecBlurb");

  const statVisitsEl = document.getElementById("tcinemaStatVisits");
  const statOpenedEl = document.getElementById("tcinemaStatOpened");
  const statFavsEl   = document.getElementById("tcinemaStatFavourites");

  let allVideos      = [];
  let filtered       = [];
  let rendered       = 0;
  let currentSort    = "newest";
  let currentVideoId = null;

  let favMap         = loadFavourites();
  let availableTags  = new Set();

  let glitchyButtons = [];
  let memory         = loadMemory();

  // bump visit count
  memory.visits = (memory.visits || 0) + 1;
  memory.last_visit_iso = new Date().toISOString();
  saveMemory();
  renderStats();

  function normalise(str) {
    return (str || "").toString().toLowerCase();
  }

  // ----------------------- Glitchy / Crown memory --------------------------
  function loadMemory() {
    try {
      var raw = window.localStorage.getItem(CROWN_KEY);
      if (!raw) return {};
      var obj = JSON.parse(raw);
      if (obj && typeof obj === "object") return obj;
    } catch (e) {}
    return {};
  }
  function saveMemory() {
    try {
      window.localStorage.setItem(CROWN_KEY, JSON.stringify(memory));
    } catch (e) {}
  }
  function renderStats() {
    if (statVisitsEl) {
      statVisitsEl.textContent = memory.visits || 0;
    }
    if (statOpenedEl) {
      statOpenedEl.textContent = memory.opened_count || 0;
    }
    if (statFavsEl) {
      statFavsEl.textContent = memory.fav_count || 0;
    }
  }
  function recordView(video) {
    if (!video || !video.id) return;
    if (!memory.viewed) memory.viewed = {};
    var id = String(video.id);
    memory.viewed[id] = (memory.viewed[id] || 0) + 1;
    memory.opened_count = Object.keys(memory.viewed).length;
    saveMemory();
    renderStats();
  }
  function recordQMode(modeId) {
    if (!modeId) return;
    if (!memory.q_counts) memory.q_counts = {};
    memory.q_counts[modeId] = (memory.q_counts[modeId] || 0) + 1;
    saveMemory();
  }

  // --------------------- Age / soul-pact gate ------------------------------
  function applyContractState(accepted) {
    if (consoleRoot) {
      consoleRoot.classList.toggle("is-unlocked", !!accepted);
    }
  }
  function initContractGate() {
    applyContractState(false);
    if (!contractEnterEl) return;

    contractEnterEl.addEventListener("click", function () {
      if (!contractAgeEl || !contractConsentEl || !contractQueensEl) return;

      if (!contractAgeEl.checked || !contractConsentEl.checked || !contractQueensEl.checked) {
        if (contractErrorEl) {
          contractErrorEl.textContent =
            "🌀🛰️⟦ᛉᛞᛟ⟧ Ｑ：ＷＩＬＬ　ＹＯＵ　ＴＲＹ　ＴＯ　ＳＷＥＡＲ　ＷＩＴＨＯＵＴ　ＢＥＩＮＧ　ＲＥＡＤＹ？ 🔴⚠️⛔ SYSTEM: all three clauses must be true before the doors unlock. 🛰️⟦ᛉᛞᛟ⟧🌀";
        }
        return;
      }

      if (contractErrorEl) contractErrorEl.textContent = "";
      applyContractState(true);
    });
  }

  // ----------------------------- Time helpers ------------------------------
  function getTimestamp(video) {
    if (!video) return 0;
    if (video.published_ts != null && video.published_ts !== "") {
      var n = Number(video.published_ts);
      if (!isNaN(n) && n > 0) return n;
    }
    if (video.published_iso) {
      var t = Date.parse(video.published_iso);
      if (!isNaN(t)) return Math.floor(t / 1000);
    }
    return 0;
  }

  function getBaseDate(video) {
    if (!video) return null;

    if (video.published_iso) {
      var parts = String(video.published_iso).split("-");
      if (parts.length >= 3) {
        var y = parseInt(parts[0],10);
        var m = parseInt(parts[1],10);
        var d = parseInt(parts[2],10);
        if (!isNaN(y) && !isNaN(m) && !isNaN(d)) {
          return new Date(Date.UTC(y,m-1,d));
        }
      }
    }

    var ts = getTimestamp(video);
    if (ts > 0) {
      var dTs = new Date(ts * 1000);
      if (!isNaN(dTs.getTime())) return dTs;
    }
    return null;
  }

  function formatMythDateParts(video) {
    var d = getBaseDate(video);
    if (!d) {
      return { short:"date unknown", long:"date unknown" };
    }

    var year  = d.getUTCFullYear();
    var month = d.getUTCMonth();
    var day   = d.getUTCDate();

    var monthNames = [
      "January","February","March","April","May","June",
      "July","August","September","October","November","December"
    ];
    var monthName = monthNames[month] || "";

    var mythYearText;
    if (year >= 2025) {
      var mcYear = year - 2024;
      mythYearText = String(mcYear).padStart(4,"0") + " MC";
    } else {
      var pmcYear = 2025 - year;
      mythYearText = "-" + String(pmcYear).padStart(4,"0") + " PMC";
    }

    var shortStr = monthName + " " + day + ", " + mythYearText;
    var longStr  = shortStr + " (" + year + " CE)";
    return { short: shortStr, long: longStr };
  }

  function formatMythDateShort(video) {
    return formatMythDateParts(video).short;
  }
  function formatMythDateLong(video) {
    return formatMythDateParts(video).long;
  }

  function compareVideosOldest(a,b) {
    var ta = getTimestamp(a);
    var tb = getTimestamp(b);
    if (ta !== tb) return ta - tb;
    return (a._index || 0) - (b._index || 0);
  }

  function sortVideos(list) {
    var copy = list.slice();
    copy.sort(function(a,b){
      return currentSort === "newest"
        ? compareVideosOldest(b,a)
        : compareVideosOldest(a,b);
    });
    return copy;
  }

  function buildEmbedUrl(video) {
    if (video && video.name && video.claim_id) {
      return "https://odysee.com/$/embed/"
        + encodeURIComponent(video.name) + "/"
        + encodeURIComponent(video.claim_id);
    }
    if (video && video.odysee_url) {
      return video.odysee_url.replace("https://odysee.com/","https://odysee.com/$/embed/");
    }
    return "";
  }

  // ---------------------------- FAVOURITES ---------------------------------
  function loadFavourites() {
    try {
      var raw = window.localStorage.getItem(FAV_KEY);
      if (!raw) return {};
      var obj = JSON.parse(raw);
      if (obj && typeof obj === "object") return obj;
    } catch (e) {}
    return {};
  }
  function saveFavourites() {
    try {
      window.localStorage.setItem(FAV_KEY, JSON.stringify(favMap));
    } catch (e) {}
    // mirror into Crown memory
    memory.fav_ids = favMap;
    memory.fav_count = Object.keys(favMap).length;
    saveMemory();
    renderStats();
  }
  function isFavourite(id) {
    return !!favMap[id];
  }
  function toggleFavourite(id) {
    if (!id) return;
    if (favMap[id]) {
      delete favMap[id];
    } else {
      favMap[id] = true;
    }
    saveFavourites();
  }
  function updateFavVisual(el, fav) {
    if (!el) return;
    el.textContent = fav ? "★" : "☆";
    el.classList.toggle("is-fav", !!fav);
  }

  // ----------------------------- FILTERS -----------------------------------
  function buildFiltersFromMetadata(videos) {
    availableTags = new Set();

    // Tag filter
    if (filterTagEl) {
      videos.forEach(function (v) {
        if (Array.isArray(v.tags)) {
          v.tags.forEach(function (t) {
            if (t) availableTags.add(t);
          });
        }
      });
      filterTagEl.innerHTML = "";
      var optAll = document.createElement("option");
      optAll.value = "";
      optAll.textContent = "All tags";
      filterTagEl.appendChild(optAll);
      Array.from(availableTags).sort().forEach(function (t) {
        var o = document.createElement("option");
        o.value = t;
        o.textContent = t;
        filterTagEl.appendChild(o);
      });
    }

    // Year filter
    if (filterYearEl) {
      var years = new Set();
      videos.forEach(function (v) {
        if (v.year) years.add(String(v.year));
      });
      filterYearEl.innerHTML = "";
      var optAllY = document.createElement("option");
      optAllY.value = "";
      optAllY.textContent = "All years";
      filterYearEl.appendChild(optAllY);
      Array.from(years).sort().reverse().forEach(function (y) {
        var o = document.createElement("option");
        o.value = y;
        o.textContent = y;
        filterYearEl.appendChild(o);
      });
    }

    // Length filter
    if (filterLengthEl) {
      filterLengthEl.innerHTML = "";
      [
        ["", "All lengths"],
        ["short", "Under 10 min"],
        ["medium", "10–30 min"],
        ["long", "Over 30 min"]
      ].forEach(function (pair) {
        var o = document.createElement("option");
        o.value = pair[0];
        o.textContent = pair[1];
        filterLengthEl.appendChild(o);
      });
    }
  }

  function initFilters() {
    [filterTagEl, filterYearEl, filterLengthEl].forEach(function (el) {
      if (!el) return;
      el.addEventListener("change", applyFilters);
    });
    if (filterFavEl) {
      filterFavEl.addEventListener("change", applyFilters);
    }
  }

  // ------------------------------- CORE RENDER -----------------------------
  function updateCount() {
    if (!countEl) return;
    var n = filtered.length;
    countEl.textContent = n === 1 ? "1 transmission" : (n + " transmissions");
  }

  function clearGrid() {
    if (gridEl) gridEl.innerHTML = "";
    rendered = 0;
  }

  function highlightActiveCard() {
    if (!gridEl) return;
    var cards = gridEl.querySelectorAll(".tcinema-card");
    cards.forEach(function(card){
      if (card.dataset.id === currentVideoId) {
        card.classList.add("is-active");
      } else {
        card.classList.remove("is-active");
      }
    });
  }

  // Glitchy recommendations --------------------------------------------------
  function computeRecommendations(sourceVideo, count) {
    if (!sourceVideo || !allVideos.length) return [];
    var baseId    = sourceVideo.id;
    var baseTags  = new Set(Array.isArray(sourceVideo.tags) ? sourceVideo.tags : []);
    var baseYear  = sourceVideo.year || null;
    var baseDur   = Number(sourceVideo.duration || 0);
    var baseTs    = getTimestamp(sourceVideo);

    function durBucket(sec) {
      var d = Number(sec || 0);
      if (d <= 10*60) return "short";
      if (d <= 30*60) return "medium";
      return "long";
    }
    var baseBucket = durBucket(baseDur);

    var scored = [];

    allVideos.forEach(function (v) {
      if (!v || v.id === baseId) return;
      var score = 0;

      // shared tags
      if (baseTags.size && Array.isArray(v.tags) && v.tags.length) {
        var shared = 0;
        v.tags.forEach(function (t) {
          if (baseTags.has(t)) shared++;
        });
        if (shared) score += shared * 3;
      }

      // same year
      if (baseYear && v.year && v.year === baseYear) {
        score += 2;
      }

      // similar length bucket
      if (durBucket(v.duration) === baseBucket) {
        score += 2;
      }

      // nearby in time (± 7 days)
      var ts = getTimestamp(v);
      if (baseTs && ts) {
        var dt = Math.abs(ts - baseTs);
        if (dt <= 7 * 24 * 3600) {
          score += 1;
        }
      }

      if (score > 0) {
        score += Math.random() * 0.5; // tiny jitter
        scored.push({ v: v, score: score });
      }
    });

    if (!scored.length) {
      // fallback: neighbouring reels in index
      var baseIndex = sourceVideo._index || 0;
      var neighbours = [];
      for (var offset = -3; offset <= 3; offset++) {
        if (offset === 0) continue;
        var idx = baseIndex + offset;
        if (idx >= 0 && idx < allVideos.length) {
          neighbours.push(allVideos[idx]);
        }
      }
      return neighbours.slice(0, count);
    }

    scored.sort(function(a,b){ return b.score - a.score; });
    return scored.slice(0, count).map(function (x) { return x.v; });
  }

  function updateRecommendations(video) {
    if (!recListEl || !recBlurbEl) return;
    recListEl.innerHTML = "";

    if (!video) {
      recBlurbEl.textContent = "Pick a reel to let Glitchy suggest the next steps in your syllabus.";
      return;
    }

    var recs = computeRecommendations(video, 4);
    if (!recs.length) {
      recBlurbEl.textContent = "Glitchy is still mapping this corridor. Try another reel or soften your filters.";
      return;
    }

    var d = Number(video.duration || 0);
    var comment;
    if (d >= 30*60) {
      comment = "You chose a deep arc. Glitchy lines up neighbouring constellations:";
    } else if (d <= 10*60) {
      comment = "Quick candle chosen. Here are a few more short sparks to follow:";
    } else {
      comment = "A mid-length walk. Glitchy threads nearby lessons into a small weave:";
    }
    recBlurbEl.textContent = comment;

    recs.forEach(function (v) {
      var card = document.createElement("button");
      card.type = "button";
      card.className = "tcinema-rec-card";
      card.dataset.id = v.id;

      var t = document.createElement("div");
      t.className = "tcinema-rec-title";
      t.textContent = v.title || "Untitled transmission";

      var m = document.createElement("div");
      m.className = "tcinema-rec-meta";
      m.textContent = formatMythDateShort(v) + " • Reel " + (v._index_display || "?");

      card.appendChild(t);
      card.appendChild(m);

      card.addEventListener("click", function () {
        setNowPlaying(v);
        highlightActiveCard();
        if (playerShell && playerShell.scrollIntoView) {
          playerShell.scrollIntoView({ behavior:"smooth", block:"start" });
        }
      });

      recListEl.appendChild(card);
    });
  }

  function setNowPlaying(video) {
    if (!video) return;
    currentVideoId = video.id;

    if (nowTitleEl) {
      nowTitleEl.textContent = video.title || "Untitled transmission";
    }
    if (nowMetaEl) {
      nowMetaEl.textContent = "Recorded • " + formatMythDateLong(video);
    }

    var src = buildEmbedUrl(video);
    if (playerFrame) {
      playerFrame.src = src || "";
    }

    if (openLinkEl) {
      var mainUrl = "#";
      if (video.odysee_url) {
        mainUrl = video.odysee_url;
      } else if (video.name) {
        mainUrl = "https://odysee.com/" + encodeURIComponent(video.name);
      }
      openLinkEl.href = mainUrl;
    }

    recordView(video);
    updateRecommendations(video);
  }

  function renderNextBatch() {
    if (!gridEl) return;
    if (rendered >= filtered.length) {
      loadMoreBtn.disabled = true;
      loadMoreBtn.textContent = "All reels loaded";
      if (statusEl) {
        statusEl.textContent = "Showing " + filtered.length + " / " + allVideos.length + " reels.";
      }
      return;
    }

    var start = rendered;
    var end   = Math.min(rendered + BATCH_SIZE, filtered.length);
    var slice = filtered.slice(start,end);

    slice.forEach(function(video){
      var card = document.createElement("article");
      card.className = "tcinema-card";
      card.dataset.id = video.id;

      var img = document.createElement("img");
      img.className = "tcinema-thumb";
      img.loading = "lazy";
      img.alt = video.title || "TCinema reel";
      img.src = video.thumbnail || video.thumb_local || video.thumb_remote || "";

      img.addEventListener("error", function () {
        if (video.thumb_remote && img.src !== video.thumb_remote) {
          img.src = video.thumb_remote;
        }
      });

      var favBtn = document.createElement("button");
      favBtn.type = "button";
      favBtn.className = "tcinema-fav-btn";
      updateFavVisual(favBtn, isFavourite(video.id));
      favBtn.addEventListener("click", function (ev) {
        ev.stopPropagation();
        toggleFavourite(video.id);
        updateFavVisual(favBtn, isFavourite(video.id));
        if (filterFavEl && filterFavEl.checked) {
          applyFilters();
        }
      });

      var body = document.createElement("div");
      body.className = "tcinema-card-body";

      var titleEl = document.createElement("h4");
      titleEl.className = "tcinema-card-title";
      titleEl.textContent = video.title || "Untitled transmission";

      var metaEl = document.createElement("div");
      metaEl.className = "tcinema-card-meta";

      var left = document.createElement("span");
      left.textContent = formatMythDateShort(video);

      var right = document.createElement("span");
      right.textContent = "Reel " + (video._index_display || "?");

      metaEl.appendChild(left);
      metaEl.appendChild(right);

      body.appendChild(titleEl);
      body.appendChild(metaEl);

      card.appendChild(img);
      card.appendChild(favBtn);
      card.appendChild(body);

      card.addEventListener("click", function () {
        setNowPlaying(video);
        highlightActiveCard();
        if (playerShell && playerShell.scrollIntoView) {
          playerShell.scrollIntoView({ behavior:"smooth", block:"start" });
        }
      });

      gridEl.appendChild(card);
    });

    rendered = end;
    if (statusEl) {
      statusEl.textContent = "Showing " + rendered + " / " + filtered.length + " reels.";
    }
  }

  function applyFilters() {
    var query   = normalise(searchInput ? searchInput.value : "");
    var tagVal  = filterTagEl ? filterTagEl.value : "";
    var yearVal = filterYearEl ? filterYearEl.value : "";
    var lenVal  = filterLengthEl ? filterLengthEl.value : "";
    var favOnly = filterFavEl && filterFavEl.checked;

    var base  = sortVideos(allVideos);

    filtered = base.filter(function (v) {
      if (query && normalise(v.title).indexOf(query) === -1) return false;

      if (tagVal) {
        if (!Array.isArray(v.tags) || v.tags.indexOf(tagVal) === -1) return false;
      }

      if (yearVal) {
        if (!v.year || String(v.year) !== String(yearVal)) return false;
      }

      if (lenVal) {
        var d = Number(v.duration || 0);
        if (lenVal === "short" && d >= 10*60) return false;
        if (lenVal === "medium" && (d < 10*60 || d > 30*60)) return false;
        if (lenVal === "long" && d <= 30*60) return false;
      }

      if (favOnly && !isFavourite(v.id)) return false;

      return true;
    });

    updateCount();
    clearGrid();
    if (filtered.length === 0) {
      if (statusEl) {
        statusEl.textContent = "No reels match that combination. Try softening your filters or asking Glitchy another Q.";
      }
      // wipe recs when nothing matches
      updateRecommendations(null);
      return;
    }

    setNowPlaying(filtered[0]);
    highlightActiveCard();
    loadMoreBtn.disabled = false;
    loadMoreBtn.textContent = "Load more reels";
    renderNextBatch();
    highlightActiveCard();
  }

  // -------------------- SORT, SEARCH, GLITCHY Qs ---------------------------
  function initSortButtons() {
    sortButtons.forEach(function (btn) {
      btn.addEventListener("click", function () {
        var val = btn.getAttribute("data-sort");
        if (!val || val === currentSort) return;
        currentSort = val;
        sortButtons.forEach(function (b) {
          b.classList.toggle("is-active", b === btn);
        });
        applyFilters();
      });
    });
  }

  function initSearch() {
    if (!searchInput) return;
    var debounce;
    searchInput.addEventListener("input", function () {
      if (debounce) window.clearTimeout(debounce);
      debounce = window.setTimeout(applyFilters, 180);
    });
  }

  function resetAllFilters() {
    if (searchInput) searchInput.value = "";
    if (filterTagEl) filterTagEl.value = "";
    if (filterYearEl) filterYearEl.value = "";
    if (filterLengthEl) filterLengthEl.value = "";
    if (filterFavEl) filterFavEl.checked = false;
  }

  function chooseAvailableTag(candidates) {
    for (var i = 0; i < candidates.length; i++) {
      var t = candidates[i];
      if (availableTags.has(t)) return t;
    }
    return "";
  }

  function showGlitchySigil(def) {
    if (!def || !glitchySigilEl) return;
    var line1 = "🌀🛰️⟦ᛉᛞᛟ⟧ " + def.q + " " + def.system + " 🛰️⟦ᛉᛞᛟ⟧🌀";
    var line2 = "▒▒🛰️⟦ ＴＲＡＮＳＭＩＳＳＩＯＮ　ＲＥＣＥＩＶＥＤ ⟧📡▒▒";
    glitchySigilEl.textContent = line1 + "\n" + line2;
  }

  function applyGlitchyMode(mode) {
    if (!mode) return;
    resetAllFilters();
    recordQMode(mode);

    if (mode === "origin") {
      currentSort = "oldest";
    } else if (mode === "latest") {
      currentSort = "newest";
    } else if (mode === "quick") {
      currentSort = "newest";
      if (filterLengthEl) filterLengthEl.value = "short";
    } else if (mode === "deep") {
      currentSort = "oldest";
      if (filterLengthEl) filterLengthEl.value = "long";
    } else if (mode === "myth") {
      currentSort = "oldest";
      var mythTag = chooseAvailableTag(["mythology","myth","Storytelling","romance","love"]);
      if (mythTag && filterTagEl) filterTagEl.value = mythTag;
    } else if (mode === "roulette") {
      currentSort = "newest";
    }

    sortButtons.forEach(function (b) {
      var val = b.getAttribute("data-sort");
      b.classList.toggle("is-active", val === currentSort);
    });

    applyFilters();

    if (mode === "roulette" && filtered.length > 0) {
      var idx = Math.floor(Math.random() * filtered.length);
      setNowPlaying(filtered[idx]);
      currentVideoId = filtered[idx].id;
      highlightActiveCard();
      if (playerShell && playerShell.scrollIntoView) {
        playerShell.scrollIntoView({ behavior:"smooth", block:"start" });
      }
    }
  }

  function renderGlitchyRail() {
    if (!glitchyRailEl) return;
    glitchyRailEl.innerHTML = "";
    glitchyButtons = [];

    var pool = GLITCHY_MODES.slice();
    for (var i = pool.length - 1; i > 0; i--) {
      var j = Math.floor(Math.random() * (i + 1));
      var tmp = pool[i];
      pool[i] = pool[j];
      pool[j] = tmp;
    }

    var max = Math.min(pool.length, 6);
    for (var k = 0; k < max; k++) {
      var def = pool[k];
      var btn = document.createElement("button");
      btn.type = "button";
      btn.className = "tcinema-glitchy-btn";
      btn.setAttribute("data-mode", def.mode);
      btn.textContent = "❓ " + def.q;
      glitchyRailEl.appendChild(btn);
      glitchyButtons.push({ el: btn, def: def });
    }

    glitchyButtons.forEach(function (pair) {
      pair.el.addEventListener("click", function () {
        applyGlitchyMode(pair.def.mode);
        showGlitchySigil(pair.def);
      });
    });
  }

  function initGlitchy() {
    renderGlitchyRail();
  }

  // ---------------------------- DATA LOAD ----------------------------------
  function loadData() {
    fetch(DATA_URL, { cache:"no-store" })
      .then(function (res) { return res.json(); })
      .then(function (data) {
        if (!Array.isArray(data)) throw new Error("tcinema-data.json must be an array");
        allVideos = data;
        if (statusEl) {
          statusEl.textContent = "Archive synced: " + allVideos.length + " reels detected.";
        }
        buildFiltersFromMetadata(allVideos);
        initFilters();
        applyFilters();
      })
      .catch(function (err) {
        console.error("TCinema load error", err);
        if (errorEl) {
          errorEl.textContent = "🌀 TCinema could not load the archive. Ring the Signal Bell above to sync from sql.json.";
        }
      });
  }

  // -------------------------- BOOT SEQUENCE --------------------------------
  initContractGate();

  if (!gridEl || !loadMoreBtn || !statusEl) return;

  loadMoreBtn.addEventListener("click", renderNextBatch);
  initSortButtons();
  initSearch();
  initGlitchy();
  loadData();
})();
</script>
<?php
$console_body_html = ob_get_clean();
require __DIR__ . '/tshell.php';
