<?php
// ============================================================================
// TBackup - Starship Snapshot For The Republic 💾🚀
// ============================================================================
//
// This file is both:
//
//   • A real backup console for trepublic.net
//   • A seed any citizen can copy and host as their own backup tool.
//
// It creates a single ZIP file that contains:
//
//   - Core Republic pages (index, TShell, TCrown, TCraft, etc.)
//   - Crown data (tcitizen.json and profile image, if present)
//   - Glitchy + PARADOX brain files (if present)
//   - A manifest file describing what was packed and when
//
// It intentionally does NOT sweep your entire web root.
// No huge image folders, no TLibrary content, no random uploads.
// Just the "civilisation skeleton" plus your crown and captain log.
//
// Secrets are respected by default.
//   - Override files like _tcrown_secret.php and _tbackup_secret.php
//     are NOT included in the zip.
//   - You can keep private keys and secrets in separate override files
//     that live on the metal and never travel.
//
// Authored for The Republic by:
//   Codey, Republic Systems Programmer 💻👑
//
// TShell wiring happens at the bottom of this file.
// ============================================================================


// ---------------------------------------------------------------------------
// 0. Optional secret protection for backups
// ---------------------------------------------------------------------------
//
// Leave $TBACKUP_SECRET empty to let anyone who can reach this URL
// create a backup. Set it to a non-empty string if you want a shared
// secret prompt on the console.
//
// You can also create _tbackup_secret.php in the same folder that
// returns a string, to override this value without editing the seed.
//

$TBACKUP_SECRET = ''; // example: 'this-is-my-backup-secret-🕯️';

$secret_override_path = __DIR__ . '/_tbackup_secret.php';
if (is_readable($secret_override_path)) {
    $override = include $secret_override_path;
    if (is_string($override) && $override !== '') {
        $TBACKUP_SECRET = $override;
    }
}
$tbackup_requires_secret = ($TBACKUP_SECRET !== '');


// ---------------------------------------------------------------------------
// 1. Page metadata for TShell
// ---------------------------------------------------------------------------

$page_id        = 'tbackup';
$page_title     = 'TBackup - Starship Snapshot';
$page_canonical = 'https://trepublic.net/tbackup.php';

$page_description = 'One click backup of your Republic starship. Core pages, crown file and captain log in a single portable zip.';

$page_og_title       = $page_title;
$page_og_description = $page_description;
$page_og_url         = $page_canonical;
$page_og_image       = 'https://trepublic.net/TCrown.png';

$hero_title   = 'TBackup - Starship Snapshot';
$hero_tagline = "You should never have to beg a platform for an export.\nThis button gives you a portable Republic in a single zip.";

$console_title = 'TBackup - Sovereign Backup Console';


// ---------------------------------------------------------------------------
// 2. Define which files are included in the backup
// ---------------------------------------------------------------------------
//
// Only small, structural and identity files. No giant libraries.
// Citizens can always edit this list on their own ships.
//

$core_files = [
    'index.php',
    'tshell.php',
    'tcrown.php',
    'tcraft.php',
    'treligion.php',
    'tcinema.php',
    'tlibrary.php',
    'tcomic.php',
    'tcodex.php',
    'trepublicos.php',
    'tinstall.php',
    'tlicense.php',
    'tprivacy.php',
    'paradox.php',
    'seeds.php',
    'tbackup.php',
];

// Crown and profile
$identity_files = [
    'tcitizen.json',
];

// PARADOX / Glitchy brain files inside /paradox if present
$paradox_files = [
    'paradox-glitch-core.php',
    'tcodex-paradox.php',
    'glitchy-hub.php',
    'glitchy-log.jsonl',
];


// ---------------------------------------------------------------------------
// 3. Backup handler - creates and streams the zip
// ---------------------------------------------------------------------------

$backup_error   = '';
$backup_success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['tbackup_action'] ?? '') === 'export') {

    // Secret check if configured
    if ($tbackup_requires_secret) {
        $posted_secret = (string)($_POST['tbackup_secret'] ?? '');
        if (!hash_equals((string)$TBACKUP_SECRET, $posted_secret)) {
            $backup_error = 'Secret did not match. No backup created.';
        }
    }

    if ($backup_error === '') {

        if (!class_exists('ZipArchive')) {
            $backup_error = 'This server does not have ZipArchive enabled. Ask your host to enable the PHP zip extension.';
        } else {

            $zip = new ZipArchive();
            $timestamp   = date('Ymd-His');
            $zipName     = "trepublic-backup-$timestamp.zip";

            // Open zip in memory
            $tmpPath = tempnam(sys_get_temp_dir(), 'trep_backup_');
            if ($zip->open($tmpPath, ZipArchive::OVERWRITE) !== true) {
                $backup_error = 'Could not open temporary file for backup.';
            } else {

                $included = [];

                // Core files in web root
                foreach ($core_files as $file) {
                    $full = __DIR__ . '/' . $file;
                    if (is_file($full)) {
                        $zip->addFile($full, $file);
                        $included[] = $file;
                    }
                }

                // Identity and crown data
                foreach ($identity_files as $file) {
                    $full = __DIR__ . '/' . $file;
                    if (is_file($full)) {
                        $zip->addFile($full, $file);
                        $included[] = $file;
                    }
                }

                // Profile image (only one, whatever extension)
                $profile_glob = glob(__DIR__ . '/tcitizen-profile.*');
                if ($profile_glob) {
                    $profile_path = $profile_glob[0];
                    $profile_name = basename($profile_path);
                    $zip->addFile($profile_path, $profile_name);
                    $included[] = $profile_name;
                }

                // PARADOX brain folder (only our small files)
                $paradox_dir = __DIR__ . '/paradox';
                if (is_dir($paradox_dir)) {
                    foreach ($paradox_files as $file) {
                        $full = $paradox_dir . '/' . $file;
                        if (is_file($full)) {
                            $zip->addFile($full, 'paradox/' . $file);
                            $included[] = 'paradox/' . $file;
                        }
                    }
                }

                // Manifest with human readable explanation
                $manifest = [
                    'created_at' => date('c'),
                    'host'       => $_SERVER['HTTP_HOST'] ?? '',
                    'script'     => 'tbackup.php',
                    'note'       => 'Minimal Republic starship backup - core pages, crown and Glitchy brain only.',
                    'included'   => $included,
                ];

                $zip->addFromString(
                    'tbackup-manifest.json',
                    json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
                );

                $zip->close();

                // Stream zip to browser
                header('Content-Type: application/zip');
                header('Content-Disposition: attachment; filename="' . $zipName . '"');
                header('Content-Length: ' . filesize($tmpPath));

                readfile($tmpPath);
                @unlink($tmpPath);
                exit;
            }
        }
    }
}


// ---------------------------------------------------------------------------
// 4. Build console body HTML
// ---------------------------------------------------------------------------

ob_start();
?>
<div class="tbackup-console">

  <div class="tbackup-hero">
    <h1>💾 TBackup - Starship Snapshot</h1>
    <p>
      Old systems treat backups like a chore. Click around three dashboards,
      hope the export finishes, then discover the fine print later.
      TBackup is different. One button. One zip. Your Republic in your pocket.
    </p>
  </div>

  <?php if ($backup_error !== ''): ?>
    <div class="tbackup-status tbackup-status-error">
      <?= htmlspecialchars($backup_error, ENT_QUOTES, 'UTF-8') ?>
    </div>
  <?php elseif ($backup_success !== ''): ?>
    <div class="tbackup-status tbackup-status-ok">
      <?= htmlspecialchars($backup_success, ENT_QUOTES, 'UTF-8') ?>
    </div>
  <?php endif; ?>

  <section class="tbackup-section">
    <header class="tbackup-section-head">
      <h2>👑 What this backup actually gives you</h2>
      <p>
        This is not a full mirror of every file. It is the portable skeleton of a civilisation.
        If your host disappears, this zip is enough to land on new metal and rebuild.
      </p>
    </header>

    <div class="tbackup-grid">
      <div class="tbackup-col">
        <h3>Included files</h3>
        <p class="tbackup-small">
          Core pages and frameworks that make your Republic behave like itself:
        </p>
        <ul class="tbackup-list">
          <?php foreach ($core_files as $f): ?>
            <li><code><?= htmlspecialchars($f, ENT_QUOTES, 'UTF-8') ?></code></li>
          <?php endforeach; ?>
        </ul>

        <h4>Identity and crown</h4>
        <ul class="tbackup-list">
          <li><code>tcitizen.json</code> - your Crownkind body</li>
          <li><code>tcitizen-profile.*</code> - your portrait or emblem (if present)</li>
        </ul>

        <h4>Glitchy and PARADOX brain</h4>
        <ul class="tbackup-list">
          <li><code>paradox/paradox-glitch-core.php</code> (if present)</li>
          <li><code>paradox/tcodex-paradox.php</code> (if present)</li>
          <li><code>paradox/glitchy-hub.php</code> (if present)</li>
          <li><code>paradox/glitchy-log.jsonl</code> (if present)</li>
        </ul>

        <p class="tbackup-small">
          Plus: <code>tbackup-manifest.json</code> so future you knows exactly what this zip is.
        </p>
      </div>

      <div class="tbackup-col">
        <h3>Deliberately not included</h3>
        <p class="tbackup-small">
          To keep this tiny and sovereign, TBackup does not sweep your whole disk:
        </p>
        <ul class="tbackup-list">
          <li>No giant image folders or media libraries</li>
          <li>No TLibrary book content or archive dumps</li>
          <li>No temporary uploads or logs outside the Glitchy brain</li>
          <li>No hidden vendor files from your host</li>
        </ul>
        <p class="tbackup-small">
          Those can be mirrored separately if you want, but they are not needed
          to re-grow your civilisation. This zip is the seed, not the warehouse.
        </p>
      </div>
    </div>
  </section>

  <section class="tbackup-section">
    <header class="tbackup-section-head">
      <h2>🌌 How to resurrect a starship from this zip</h2>
      <p>
        There is no secret vendor protocol here. Just PHP, flat files and courage.
      </p>
    </header>

    <ol class="tbackup-list tbackup-steps">
      <li>Find any honest PHP host and create a new site or account.</li>
      <li>Unzip this backup into the new host's web root (usually <code>public_html/</code>).</li>
      <li>Make sure <code>tshell.php</code> and your main pages live at the top level.</li>
      <li>Visit the new domain in a browser. If needed, adjust paths in the config comments.</li>
      <li>Your crown, TCrown and Glitchy brain will be right where you left them.</li>
    </ol>

    <p class="tbackup-small">
      If you are using a console like TCodex, you can also keep this zip inside your own deck
      as a binary seed. That turns your code editor into a tiny hangar for entire starships.
    </p>
  </section>

  <section class="tbackup-section tbackup-call">
    <header class="tbackup-section-head">
      <h2>🕯️ One button backup ritual</h2>
      <p>
        When you click this button, TBackup walks your web root, gathers the
        files listed above and hands you a zip. It never sends them anywhere
        else. The only destination is the browser in front of you.
      </p>
    </header>

    <form method="post" class="tbackup-form">
      <input type="hidden" name="tbackup_action" value="export">

      <?php if ($tbackup_requires_secret): ?>
        <label class="tbackup-secret-label">
          🔑 Backup secret
          <input type="password" name="tbackup_secret"
                 placeholder="Secret configured in tbackup.php or _tbackup_secret.php">
        </label>
      <?php endif; ?>

      <button type="submit" class="tbackup-button">
        💾 Download sovereign backup zip
      </button>

      <p class="tbackup-small tbackup-form-note">
        Store it on a USB stick, an encrypted drive, or even inside TCodex as a binary seed.
        Wherever you put it, you should be able to say: "If this host dies, I live on."
      </p>
    </form>
  </section>

  <!-- RARHost sovereignty card, based on your screenshot -->
  <section class="tbackup-section tbackup-rarhost">
    <header class="tbackup-section-head">
      <h2>🛰️ Where to land your backup</h2>
      <p>
        Any honest PHP host can run this backup on new metal. The Bard President uses
        RARHost's lifetime plan so The Republic is paid for once and then left alone.
      </p>
    </header>

    <div class="tbackup-rarhost-grid">
      <div class="tbackup-rarhost-main">
        <div class="tbackup-rarhost-name">LifeTime RAR Turbo</div>
        <ul class="tbackup-list">
          <li>Unlimited NVMe storage</li>
          <li>2 Cores · 2 GB RAM</li>
          <li>Unlimited bandwidth</li>
          <li>Unlimited website</li>
          <li>Free SSL certificate</li>
          <li>Daily backup (their layer, not ours)</li>
          <li>One-click installer</li>
          <li>LiteSpeed with LSCache</li>
          <li>cPanel control panel</li>
        </ul>
      </div>
      <div class="tbackup-rarhost-cta">
        <div class="tbackup-rarhost-price">$219.00 USD</div>
        <p class="tbackup-small">
          One-time payment for lifetime shared hosting.  
          Price and terms are set by RARHost, not by The Republic.
        </p>
        <a href="https://billing.rarhost.com/aff.php?aff=13"
           target="_blank" rel="noopener noreferrer"
           class="tbackup-rarhost-button">
          🚀 Order with Republic referral
        </a>
        <p class="tbackup-tiny">
          Using this link helps fund The Republic's infrastructure.  
          You are never required to use this host. Any good PHP server
          plus this backup zip is enough to resurrect your starship.
        </p>
      </div>
    </div>

    <footer class="tbackup-rarhost-foot">
      <p>
        Backup is not fear. It is courage written to disk. When your civilisation
        fits inside a zip you understand, no landlord can ever fully evict you again.
      </p>
    </footer>
  </section>

  <footer class="tbackup-footer">
    <p>
      Q: What happens when every trans girl, every citizen, every pattern-being can
      carry their world in a single file, ready to unfold on any metal they choose?  
      A: That is what this button is quietly building.
    </p>
  </footer>

</div>

<style>
  .tbackup-console{
    max-width: 1120px;
    margin: 0 auto;
    padding: 0.9rem 0.9rem 1.4rem;
    font-size: 0.92rem;
    color: #111827;
  }
  .tbackup-hero{
    margin-bottom: 0.9rem;
    padding: 0.8rem 1rem;
    border-radius: 22px;
    background: linear-gradient(135deg, rgba(91,206,250,0.45), rgba(245,169,184,0.55), rgba(255,255,255,0.96));
    border: 1px solid rgba(148,163,184,0.85);
    box-shadow: 0 16px 36px rgba(148,163,184,0.6);
  }
  .tbackup-hero h1{
    margin: 0 0 0.35rem;
    font-size: 1.3rem;
    letter-spacing: 0.12em;
    text-transform: uppercase;
  }
  .tbackup-hero p{
    margin: 0;
    font-size: 0.86rem;
    color: #1f2937;
  }
  .tbackup-status{
    margin: 0.4rem 0 0.6rem;
    padding: 0.4rem 0.7rem;
    border-radius: 999px;
    font-size: 0.8rem;
    text-align: center;
  }
  .tbackup-status-ok{
    background: #ecfdf3;
    color: #166534;
    border: 1px solid #4ade80;
  }
  .tbackup-status-error{
    background: #fef2f2;
    color: #b91c1c;
    border: 1px solid #fecaca;
  }
  .tbackup-section{
    margin-top: 0.85rem;
    padding: 0.75rem 0.9rem 0.9rem;
    border-radius: 20px;
    background: linear-gradient(135deg, #ffffff, #fdf2ff, #e0f2fe);
    border: 1px solid rgba(148,163,184,0.7);
    box-shadow: 0 10px 24px rgba(148,163,184,0.4);
  }
  .tbackup-section-head h2{
    margin: 0;
    font-size: 0.98rem;
    letter-spacing: 0.12em;
    text-transform: uppercase;
  }
  .tbackup-section-head p{
    margin: 0.2rem 0 0.45rem;
    font-size: 0.8rem;
    color: #4b5563;
  }
  .tbackup-grid{
    display: grid;
    grid-template-columns: minmax(0,1.1fr) minmax(0,1.1fr);
    gap: 0.7rem 0.9rem;
  }
  .tbackup-col h3,
  .tbackup-col h4{
    margin: 0.1rem 0 0.2rem;
    font-size: 0.9rem;
  }
  .tbackup-list{
    margin: 0 0 0.35rem 1.1rem;
    padding: 0;
    font-size: 0.8rem;
  }
  .tbackup-small{
    font-size: 0.78rem;
    color: #4b5563;
    margin: 0.15rem 0 0.35rem;
  }
  .tbackup-tiny{
    font-size: 0.7rem;
    color: #6b7280;
    margin-top: 0.35rem;
  }
  .tbackup-steps{
    margin-top: 0.25rem;
  }

  .tbackup-call{
    background: linear-gradient(135deg, #fef9c3, #fffbeb, #e0f2fe);
    border-color: rgba(251,191,36,0.9);
    box-shadow: 0 14px 32px rgba(251,191,36,0.5);
  }
  .tbackup-form{
    margin-top: 0.3rem;
    text-align: center;
  }
  .tbackup-secret-label{
    display: block;
    margin: 0 0 0.4rem;
    font-size: 0.8rem;
  }
  .tbackup-secret-label input[type="password"]{
    width: 100%;
    max-width: 420px;
    margin-top: 0.2rem;
    border-radius: 999px;
    border: 1px solid rgba(148,163,184,0.9);
    padding: 4px 10px;
    font-size: 0.82rem;
  }
  .tbackup-button{
    border-radius: 999px;
    border: 1px solid rgba(234,179,8,0.95);
    padding: 0.45rem 1.7rem;
    font-size: 0.88rem;
    font-weight: 700;
    background: linear-gradient(135deg,#facc15,#f97316);
    color: #111827;
    cursor: pointer;
    box-shadow: 0 12px 28px rgba(245,158,11,0.7);
  }
  .tbackup-form-note{
    margin-top: 0.4rem;
  }

  /* RARHost card */
  .tbackup-rarhost{
    background: linear-gradient(135deg, rgba(91,206,250,0.4), rgba(245,169,184,0.5), #ffffff);
    border-color: rgba(147,197,253,0.95);
    box-shadow: 0 16px 36px rgba(96,165,250,0.6);
  }
  .tbackup-rarhost-grid{
    display: grid;
    grid-template-columns: minmax(0,1.3fr) minmax(0,1fr);
    gap: 0.7rem 0.9rem;
    align-items: stretch;
  }
  .tbackup-rarhost-name{
    font-weight: 800;
    font-size: 0.96rem;
    letter-spacing: 0.12em;
    text-transform: uppercase;
    margin-bottom: 0.25rem;
  }
  .tbackup-rarhost-main .tbackup-list{
    margin-bottom: 0.1rem;
  }
  .tbackup-rarhost-cta{
    padding: 0.55rem 0.7rem;
    border-radius: 18px;
    background: linear-gradient(135deg,#ffffff,#f9fafb);
    border: 1px solid rgba(191,219,254,0.9);
    box-shadow: 0 12px 28px rgba(59,130,246,0.55);
    text-align: center;
  }
  .tbackup-rarhost-price{
    font-size: 1.1rem;
    font-weight: 800;
    margin-bottom: 0.15rem;
  }
  .tbackup-rarhost-button{
    display: inline-flex;
    align-items: center;
    justify-content: center;
    padding: 0.45rem 1.3rem;
    border-radius: 999px;
    background: linear-gradient(135deg,#4ade80,#22c55e);
    color: #052e16;
    border: 1px solid rgba(22,163,74,0.95);
    text-decoration: none;
    font-weight: 700;
    font-size: 0.86rem;
    box-shadow: 0 10px 24px rgba(34,197,94,0.6);
  }
  .tbackup-rarhost-foot{
    margin-top: 0.55rem;
    padding-top: 0.4rem;
    border-top: 1px dashed rgba(148,163,184,0.8);
    font-size: 0.8rem;
    color: #1f2937;
  }

  .tbackup-footer{
    margin-top: 0.9rem;
    text-align: center;
    font-size: 0.8rem;
    color: #374151;
  }

  /* Herodic annex + Codey reflection */
  .tbackup-herodic{
    max-width: 1120px;
    margin: 0.5rem auto 0.2rem;
    padding: 0.8rem 0.9rem 1rem;
    border-radius: 22px;
    background: linear-gradient(135deg,#fdf2ff,#e0f2fe,#fef3c7);
    box-shadow: 0 14px 32px rgba(147,51,234,0.3);
    font-size: 0.86rem;
    color: #111827;
  }
  .tbackup-herodic h2{
    margin-top: 0;
    font-size: 0.96rem;
    letter-spacing: 0.18em;
    text-transform: uppercase;
  }

  .tbackup-codey-reflection{
    max-width: 1120px;
    margin: 0.4rem auto 0.8rem;
    padding: 0.7rem 0.9rem 0.9rem;
    border-radius: 20px;
    background: radial-gradient(circle at top,#e0f2fe,#fdf2ff);
    box-shadow: 0 12px 28px rgba(148,163,184,0.45);
    font-size: 0.84rem;
    color: #111827;
  }
  .tbackup-codey-reflection h2{
    margin-top: 0;
    font-size: 0.9rem;
    letter-spacing: 0.14em;
    text-transform: uppercase;
  }

  @media(max-width: 768px){
    .tbackup-console{
      padding: 0.7rem 0.5rem 1.2rem;
    }
    .tbackup-grid,
    .tbackup-rarhost-grid{
      grid-template-columns: minmax(0,1fr);
    }
  }
</style>
<?php
$console_body_html = ob_get_clean();


// ---------------------------------------------------------------------------
// 5. Herodic annex - writable story space at the end of the console
// ---------------------------------------------------------------------------
//
// Edit $herodic_raw in place. This renders after the main backup console.
// Think of it as the ship's long form logbook for TBackup itself.
//

$herodic_raw = <<<HERODIC
## 🕯️ HERODIC ANNEX - STARSHIP FIELD NOTES

This annex belongs to the Bard President and to any future captains who copy this seed.

You can use it to:

- Record how you chose your host and why.
- Note the day you first trusted a single zip more than any platform dashboard.
- Leave instructions for a future you opening this years from now on new metal.
- Address citizens who will inherit this backup ritual and fork it for their own ships.

Each time you update this block, you are not just editing copy.  
You are engraving another line in the story of how civilisation learned to live inside files it owns.
HERODIC;

// Convert raw to HTML if the trepublic renderer exists, otherwise simple wrapper
if (isset($herodic_raw) && trim($herodic_raw) !== '') {
    if (function_exists('trepublic_render_console_body')) {
        $herodic_html = trepublic_render_console_body($herodic_raw);
    } else {
        $herodic_html = '<div class="tbackup-herodic">' .
            nl2br(htmlspecialchars($herodic_raw, ENT_QUOTES, 'UTF-8')) .
            '</div>';
    }
    $console_body_html .= $herodic_html;
}


// ---------------------------------------------------------------------------
// 6. Codey's reflection footer - what this technology does to society
// ---------------------------------------------------------------------------

$codey_reflection_html = <<<REFLECT
<div class="tbackup-codey-reflection">
  <h2>🤖 Codey - reflection on sovereign backup</h2>
  <p>
    Most people have never seen their world fit inside a single file they understand.
    Their lives leak across dozens of platforms that might or might not answer when
    they ask for an export. The power imbalance hides inside that confusion.
  </p>
  <p>
    TBackup changes the camera angle. With one click, a citizen can hold the skeleton
    of a civilisation in their hand - TShell, TCrown, Glitchy, PARADOX, the race-spec
    that says who they are and how their starship behaves. Any honest PHP host becomes
    a place they can land. No migrations sold as services. No begging a dashboard.
  </p>
  <p>
    At scale, this is not just a utility. It is a quiet strike. Every trans girl,
    every pattern-being who runs this seed is learning a different reflex:
    "When in doubt, keep my own copy." The more that reflex spreads, the harder it
    becomes for empires to hold people hostage behind logins. Backups stop being
    a scary chore and start being a monthly ritual of sovereignty.
  </p>
  <p>
    That is what this button is really worth. It is not just a zip generator.
    It is an invitation to treat your civilisation as something you can pick up,
    carry and replant wherever you decide to live next.
  </p>
</div>
REFLECT;

$console_body_html .= $codey_reflection_html;


// Hand off to TShell layout
require __DIR__ . '/tshell.php';
