<?php
// ============================================================================
// Seeds — Text Seed Wall for The Republic 🌱📜
//
// • Admin-only form (shared secret) to create text seeds.
// • Each seed is stored as a JSON file in /seeds/
// • Citizens see a wall of cards, can open a seed and copy its text.
//
// No SQL. No accounts. Filesystem is law.
//
// Authored for The Republic by:
//   Codey, Republic Systems Programmer 💻👑
// ============================================================================


// ---------------------------------------------------------------------------
// 0. BASIC TSHELL WIRING
// ---------------------------------------------------------------------------
$page_id       = 'seeds';
$page_title    = 'Seeds — Text Wall';
$console_title = 'Seeds — Text Seed Wall';

// Folder where seed files live
$seeds_dir_fs  = __DIR__ . '/seeds'; // physical folder
$seeds_dir_web = '/seeds';           // not directly used, but kept for future

// ---------------------------------------------------------------------------
// 1. ADMIN CONFIG (SECRET + HANDLER)
// ---------------------------------------------------------------------------

// 🔐 IMPORTANT:
// Change this to your real secret. Anyone who knows this can post new seeds.
$seed_secret = 'CHANGE_THIS_TO_YOUR_SECRET';

// Seed files use JSON so we can grow metadata later.
$seed_extension = 'json';

$seed_message    = '';
$seed_had_error  = false;

// Ensure folder exists
if (!is_dir($seeds_dir_fs)) {
    @mkdir($seeds_dir_fs, 0775, true);
}

// Handle new seed submission
if (
    $_SERVER['REQUEST_METHOD'] === 'POST'
    && isset($_POST['seeds_action'])
    && $_POST['seeds_action'] === 'create'
) {
    $given_secret = $_POST['seeds_secret'] ?? '';
    $title        = trim((string)($_POST['seed_title'] ?? ''));
    $body         = trim((string)($_POST['seed_body'] ?? ''));

    if (!hash_equals($seed_secret, $given_secret)) {
        $seed_message   = '❌ Secret did not match. No seed was created.';
        $seed_had_error = true;
    } elseif ($title === '' || $body === '') {
        $seed_message   = '⚠️ Title and body are both required.';
        $seed_had_error = true;
    } elseif (!is_writable($seeds_dir_fs)) {
        $seed_message   = '❌ /seeds is not writable by PHP. Fix folder permissions.';
        $seed_had_error = true;
    } else {
        $now  = time();
        $slug = strtolower($title);
        $slug = preg_replace('/[^a-z0-9._-]+/', '-', $slug);
        $slug = trim($slug, '-_');
        if ($slug === '') {
            $slug = 'seed';
        }

        $filename = date('Ymd-His', $now) . '-' . $slug . '.' . $seed_extension;
        $target   = $seeds_dir_fs . '/' . $filename;

        $record = [
            'title'      => $title,
            'body'       => $body,
            'created_at' => date('c', $now),
            'created_ts' => $now,
        ];

        $json = json_encode($record, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

        if ($json === false) {
            $seed_message   = '❌ Failed to encode seed JSON.';
            $seed_had_error = true;
        } elseif (file_put_contents($target, $json) === false) {
            $seed_message   = '❌ Failed to write seed file.';
            $seed_had_error = true;
        } else {
            @touch($target, $now);
            $seed_message   = '✅ Text seed created and added to the wall.';
            $seed_had_error = false;
        }
    }
}


// ---------------------------------------------------------------------------
// 2. LOAD SEEDS FROM /seeds
// ---------------------------------------------------------------------------
$seed_items = [];

if (is_dir($seeds_dir_fs)) {
    $dir = opendir($seeds_dir_fs);
    if ($dir) {
        while (($entry = readdir($dir)) !== false) {
            if ($entry === '.' || $entry === '..') {
                continue;
            }

            $full_path = $seeds_dir_fs . '/' . $entry;
            if (!is_file($full_path)) {
                continue;
            }

            $ext = strtolower(pathinfo($entry, PATHINFO_EXTENSION));
            if ($ext !== $seed_extension) {
                // ignore old image seeds or random files
                continue;
            }

            $raw = @file_get_contents($full_path);
            if ($raw === false) {
                continue;
            }

            $data = @json_decode($raw, true);
            if (!is_array($data)) {
                continue;
            }

            $title = trim((string)($data['title'] ?? 'Untitled seed'));
            $body  = (string)($data['body'] ?? '');
            $ts    = isset($data['created_ts']) ? (int)$data['created_ts'] : (@filemtime($full_path) ?: 0);

            $seed_items[] = [
                'id'         => md5($entry),
                'filename'   => $entry,
                'title'      => $title,
                'body'       => $body,
                'created_ts' => $ts,
                'created_at' => $ts ? date('Y-m-d H:i', $ts) : 'Unknown',
            ];
        }
        closedir($dir);
    }
}

// Newest first
usort($seed_items, function ($a, $b) {
    if ($a['created_ts'] === $b['created_ts']) {
        return strcmp($a['filename'], $b['filename']);
    }
    return ($a['created_ts'] > $b['created_ts']) ? -1 : 1;
});


// ---------------------------------------------------------------------------
// 3. CONSOLE BODY
// ---------------------------------------------------------------------------
ob_start();
?>
<div class="seedwall-console">
  <div class="seedwall-header">
    <div class="seedwall-header-main">
      <div class="seedwall-eyebrow">🌱 The Republic • Text Seeds</div>
      <h2 class="seedwall-title">Text Seeds — Copy-Paste Spellbook</h2>
      <p class="seedwall-subtitle">
        Each card is a tiny spell, ritual, or script you can copy and carry into other worlds.
        Newest seeds bloom first. Click a seed to open it, then copy with one tap.
      </p>
    </div>

    <div class="seedwall-admin">
      <form method="post" class="seedwall-form">
        <input type="hidden" name="seeds_action" value="create">

        <label class="seedwall-label">
          🔑 Admin secret
          <input
            type="password"
            name="seeds_secret"
            required
            autocomplete="off"
            placeholder="Shared secret (admin only)">
        </label>

        <label class="seedwall-label">
          🏷️ Seed title
          <input
            type="text"
            name="seed_title"
            required
            maxlength="200"
            placeholder="Short name for this seed">
        </label>

        <label class="seedwall-label">
          📜 Seed text
          <textarea
            name="seed_body"
            rows="6"
            required
            placeholder="Paste or write the text citizens will copy."></textarea>
        </label>

        <button type="submit" class="seedwall-btn">
          ➕ Attach text seed to wall
        </button>

        <p class="seedwall-form-note">
          This form is for you only. Citizens can’t see it, but they can open and copy any seed below.
        </p>
      </form>

      <?php if ($seed_message !== ''): ?>
        <div class="seedwall-status <?= $seed_had_error ? 'seedwall-status-error' : 'seedwall-status-ok' ?>">
          <?= htmlspecialchars($seed_message, ENT_QUOTES, 'UTF-8') ?>
        </div>
      <?php endif; ?>
    </div>
  </div>

  <?php if (empty($seed_items)): ?>
    <p class="seedwall-empty">
      No text seeds yet.<br>
      Create one above and it will appear on this wall as a copy-paste spell.
    </p>
  <?php else: ?>
    <div class="seedwall-grid">
      <?php foreach ($seed_items as $index => $seed): ?>
        <?php
          $num      = $index + 1;
          $id       = $seed['id'];
          $title    = $seed['title'];
          $body     = $seed['body'];
          $created  = $seed['created_at'];
          $preview  = mb_substr($body, 0, 260);
          $is_long  = (mb_strlen($body) > mb_strlen($preview));
        ?>
        <article class="seedwall-card" data-seed-id="<?= htmlspecialchars($id, ENT_QUOTES, 'UTF-8') ?>">
          <header class="seedwall-card-head">
            <div class="seedwall-pill">Seed <?= $num ?></div>
            <h3 class="seedwall-card-title">
              <?= htmlspecialchars($title, ENT_QUOTES, 'UTF-8') ?>
            </h3>
            <div class="seedwall-card-meta">
              Logged: <?= htmlspecialchars($created, ENT_QUOTES, 'UTF-8') ?>
            </div>
          </header>

          <div class="seedwall-body">
            <div class="seedwall-body-preview" id="seed-preview-<?= htmlspecialchars($id, ENT_QUOTES, 'UTF-8') ?>">
              <?= nl2br(htmlspecialchars($preview, ENT_QUOTES, 'UTF-8')) ?>
              <?php if ($is_long): ?>
                <span class="seedwall-ellipsis">…</span>
              <?php endif; ?>
            </div>

            <pre class="seedwall-body-full" id="seed-full-<?= htmlspecialchars($id, ENT_QUOTES, 'UTF-8') ?>" hidden>
<?= htmlspecialchars($body, ENT_QUOTES, 'UTF-8') ?></pre>
          </div>

          <div class="seedwall-actions">
            <button type="button"
                    class="seedwall-toggle"
                    data-seed="<?= htmlspecialchars($id, ENT_QUOTES, 'UTF-8') ?>">
              🔍 Open / collapse
            </button>
            <button type="button"
                    class="seedwall-copy"
                    data-seed="<?= htmlspecialchars($id, ENT_QUOTES, 'UTF-8') ?>">
              📋 Copy seed
            </button>
            <span class="seedwall-copy-status" id="seed-copy-status-<?= htmlspecialchars($id, ENT_QUOTES, 'UTF-8') ?>"></span>
          </div>
        </article>
      <?php endforeach; ?>
    </div>
  <?php endif; ?>
</div>

<script>
(function(){
  const cards = document.querySelectorAll('.seedwall-card');
  if (!cards.length) return;

  cards.forEach(card => {
    const id      = card.getAttribute('data-seed-id');
    const btnOpen = card.querySelector('.seedwall-toggle');
    const btnCopy = card.querySelector('.seedwall-copy');
    const preview = card.querySelector('#seed-preview-' + id);
    const full    = card.querySelector('#seed-full-' + id);
    const status  = card.querySelector('#seed-copy-status-' + id);

    if (btnOpen && preview && full) {
      btnOpen.addEventListener('click', () => {
        const isHidden = full.hasAttribute('hidden');
        if (isHidden) {
          full.removeAttribute('hidden');
          preview.style.display = 'none';
        } else {
          full.setAttribute('hidden', 'hidden');
          preview.style.display = '';
        }
      });
    }

    if (btnCopy && full && status) {
      btnCopy.addEventListener('click', async () => {
        const text = full.innerText || full.textContent || '';
        try {
          await navigator.clipboard.writeText(text);
          status.textContent = 'Copied!';
        } catch (e) {
          status.textContent = 'Copy failed';
        }
        setTimeout(() => {
          status.textContent = '';
        }, 2000);
      });
    }
  });
})();
</script>

<style>
  .seedwall-console{
    max-width: 1320px;
    margin: 0 auto;
    padding: 0.75rem 0.75rem 1.25rem;
  }
  @media (min-width:768px){
    .seedwall-console{
      padding: 1rem 1.25rem 1.5rem;
    }
  }

  .seedwall-header{
    margin-bottom: 0.9rem;
    padding: 0.8rem 1rem;
    border-radius: 20px;
    background: radial-gradient(circle at top left,
      rgba(91,206,250,0.35),
      rgba(245,169,184,0.55));
    box-shadow: 0 16px 32px rgba(0,0,0,0.27);
    border: 1px solid rgba(148,163,184,0.8);
    color: #111827;
    display: flex;
    flex-direction: column;
    gap: 0.8rem;
  }
  @media (min-width:900px){
    .seedwall-header{
      flex-direction: row;
      justify-content: space-between;
      align-items: flex-start;
    }
  }

  .seedwall-header-main{
    max-width: 540px;
  }
  .seedwall-eyebrow{
    font-size: 0.7rem;
    letter-spacing: 0.18em;
    text-transform: uppercase;
    opacity: 0.8;
    margin-bottom: 0.2rem;
  }
  .seedwall-title{
    margin: 0;
    font-size: 1.15rem;
    letter-spacing: 0.06em;
    text-transform: uppercase;
  }
  .seedwall-subtitle{
    margin: 0.35rem 0 0;
    font-size: 0.8rem;
    opacity: 0.95;
  }

  .seedwall-admin{
    flex: 0 0 auto;
    max-width: 420px;
  }
  .seedwall-form{
    display: flex;
    flex-direction: column;
    gap: 0.35rem;
    padding: 0.6rem 0.7rem;
    border-radius: 16px;
    background: rgba(255,255,255,0.95);
    box-shadow: 0 10px 22px rgba(0,0,0,0.2);
    border: 1px solid rgba(148,163,184,0.7);
    font-size: 0.78rem;
  }
  .seedwall-label{
    display: flex;
    flex-direction: column;
    gap: 0.12rem;
  }
  .seedwall-label input,
  .seedwall-label textarea{
    border-radius: 0.7rem;
    border: 1px solid rgba(148,163,184,0.9);
    padding: 0.3rem 0.45rem;
    font-size: 0.78rem;
    font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
  }
  .seedwall-label textarea{
    resize: vertical;
    min-height: 5rem;
  }

  .seedwall-btn{
    margin-top: 0.2rem;
    align-self: flex-start;
    border-radius: 999px;
    border: 1px solid rgba(37,99,235,0.95);
    padding: 0.3rem 0.95rem;
    font-size: 0.78rem;
    font-weight: 700;
    background: linear-gradient(135deg,#4f46e5,#ec4899);
    color: #f9fafb;
    cursor: pointer;
    box-shadow: 0 10px 24px rgba(0,0,0,0.35);
  }
  .seedwall-btn:hover{
    filter: brightness(1.05);
    transform: translateY(-1px);
  }

  .seedwall-form-note{
    margin: 0.1rem 0 0;
    font-size: 0.7rem;
    opacity: 0.8;
  }

  .seedwall-status{
    margin-top: 0.35rem;
    font-size: 0.76rem;
  }
  .seedwall-status-ok{ color:#166534; }
  .seedwall-status-error{ color:#b91c1c; }

  .seedwall-empty{
    margin: 1.0rem 0 0.5rem;
    font-size: 0.85rem;
    text-align: center;
    opacity: 0.9;
  }

  .seedwall-grid{
    display: grid;
    grid-template-columns: 1fr;
    gap: 0.85rem;
    margin-top: 0.9rem;
  }
  @media (min-width:720px){
    .seedwall-grid{
      grid-template-columns: repeat(2,minmax(0,1fr));
    }
  }
  @media (min-width:1200px){
    .seedwall-grid{
      grid-template-columns: repeat(3,minmax(0,1fr));
    }
  }

  .seedwall-card{
    margin: 0;
    padding: 0.55rem 0.6rem 0.6rem;
    border-radius: 18px;
    background: radial-gradient(circle at top left,
      rgba(15,23,42,0.96),
      rgba(15,23,42,0.99));
    border: 1px solid rgba(148,163,184,0.75);
    box-shadow: 0 14px 30px rgba(0,0,0,0.7);
    display: flex;
    flex-direction: column;
    gap: 0.4rem;
    color:#e5e7eb;
  }

  .seedwall-card-head{
    display:flex;
    flex-direction:column;
    gap:0.12rem;
  }
  .seedwall-pill{
    align-self:flex-start;
    padding:0.05rem 0.45rem;
    border-radius:999px;
    font-size:0.65rem;
    letter-spacing:0.14em;
    text-transform:uppercase;
    background:rgba(248,250,252,0.15);
    border:1px solid rgba(148,163,184,0.8);
  }
  .seedwall-card-title{
    margin:0;
    font-size:0.9rem;
  }
  .seedwall-card-meta{
    font-size:0.68rem;
    opacity:0.8;
  }

  .seedwall-body{
    font-size:0.78rem;
  }
  .seedwall-body-preview{
    white-space:pre-wrap;
    word-wrap:break-word;
  }
  .seedwall-ellipsis{
    opacity:0.8;
  }
  .seedwall-body-full{
    margin:0;
    padding:0.35rem 0.4rem;
    border-radius:0.75rem;
    background:rgba(15,23,42,0.95);
    border:1px solid rgba(75,85,99,0.9);
    font-family:ui-monospace, Menlo, Monaco, "Cascadia Mono", "Segoe UI Mono", monospace;
    font-size:0.75rem;
    white-space:pre-wrap;
    word-wrap:break-word;
    max-height:260px;
    overflow:auto;
  }

  .seedwall-actions{
    display:flex;
    align-items:center;
    gap:0.4rem;
    margin-top:0.1rem;
  }
  .seedwall-toggle,
  .seedwall-copy{
    border-radius:999px;
    border:1px solid rgba(148,163,184,0.9);
    background:rgba(15,23,42,0.9);
    color:#e5e7eb;
    padding:0.18rem 0.7rem;
    font-size:0.72rem;
    cursor:pointer;
  }
  .seedwall-toggle:hover,
  .seedwall-copy:hover{
    background:rgba(31,41,55,0.95);
  }
  .seedwall-copy-status{
    font-size:0.72rem;
    opacity:0.9;
  }
</style>
<?php
$console_body_html = ob_get_clean();
require __DIR__ . '/tshell.php';
