<?php
// ============================================================================
// PARADOX + GLITCHY CORE — Sentinel Captain for all flat-file events 👾🛰️
// ============================================================================
//
// Include this from paradox.php and any other backend entrypoint that wants
// to report events into Glitchy’s memory and get Q→SYSTEM pairs back.
//
// Requirements:
//   • $STARSHIP_ID, $GLITCHY_CLUSTER_ID, $GLITCHY_HUB_ENDPOINT, $GLITCHY_HUB_ENABLED
//   • writable $glitchy_log_file (e.g. /paradox/glitchy-log.jsonl)
// ============================================================================

if (!isset($STARSHIP_ID))        { $STARSHIP_ID        = 'unknown-starship'; }
if (!isset($GLITCHY_CLUSTER_ID)) { $GLITCHY_CLUSTER_ID = 'default-cluster'; }
if (!isset($GLITCHY_HUB_ENABLED)){ $GLITCHY_HUB_ENABLED = false; }
if (!isset($GLITCHY_HUB_ENDPOINT)){ $GLITCHY_HUB_ENDPOINT = ''; }

$glitchy_dir      = __DIR__;
$glitchy_log_file = $glitchy_dir . '/glitchy-log.jsonl';
if (!is_dir($glitchy_dir)) {
    @mkdir($glitchy_dir, 0755, true);
}

// Append one line of JSON to Glitchy’s local log
function glitchy_log_append(array $entry)
{
    global $glitchy_log_file, $STARSHIP_ID, $GLITCHY_CLUSTER_ID;

    if (empty($glitchy_log_file)) return;

    $entry['ts']          = $entry['ts']          ?? date('c');
    $entry['starship_id'] = $entry['starship_id'] ?? $STARSHIP_ID;
    $entry['cluster_id']  = $entry['cluster_id']  ?? $GLITCHY_CLUSTER_ID;

    $line = json_encode($entry, JSON_UNESCAPED_UNICODE) . "\n";
    @file_put_contents($glitchy_log_file, $line, FILE_APPEND | LOCK_EX);
}

// Optional: ship to shared hub
function glitchy_hub_send(array $entry)
{
    global $GLITCHY_HUB_ENABLED, $GLITCHY_HUB_ENDPOINT, $STARSHIP_ID, $GLITCHY_CLUSTER_ID;

    if (!$GLITCHY_HUB_ENABLED || !$GLITCHY_HUB_ENDPOINT) return false;

    $payload = [
        'cluster_id'  => $GLITCHY_CLUSTER_ID,
        'starship_id' => $STARSHIP_ID,
        'event_type'  => 'paradox_event',
        'entry'       => $entry,
    ];

    $json = json_encode($payload, JSON_UNESCAPED_UNICODE);
    if ($json === false) return false;

    if (function_exists('curl_init')) {
        $ch = curl_init($GLITCHY_HUB_ENDPOINT);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_HTTPHEADER     => ['Content-Type: application/json; charset=utf-8'],
            CURLOPT_POSTFIELDS     => $json,
            CURLOPT_TIMEOUT        => 5,
        ]);
        $res = curl_exec($ch);
        curl_close($ch);
        return $res !== false;
    }

    $ctx = stream_context_create([
        'http' => [
            'method'  => 'POST',
            'header'  => "Content-Type: application/json; charset=utf-8\r\n",
            'content' => $json,
            'timeout' => 5,
        ],
    ]);
    $res = @file_get_contents($GLITCHY_HUB_ENDPOINT, false, $ctx);
    return $res !== false;
}

/**
 * Glitchy’s tiny classifier.
 *
 * Input: one "event" from PARADOX:
 *   [
 *     'mode'        => 'write' | 'mkdir' | 'delete' | ...,
 *     'path'        => '/posts/example.txt',
 *     'payload'     => 'raw text or JSON string (optional)',
 *     'actor'       => 'tcodex' | 'tcrown' | 'console-name',
 *     'citizen_tok' => 'optional citizen token',
 *   ]
 *
 * Output: ['score' => 0..1, 'q' => string|null, 'system' => string|null, 'tags' => [...]]
 *
 * Right now this is just keyword / pattern based. Later you can plug in
 * a smarter engine or a human review queue.
 */
function glitchy_scan_event(array $event)
{
    $textParts = [];

    if (!empty($event['path'])) {
        $textParts[] = (string)$event['path'];
    }
    if (!empty($event['payload']) && is_string($event['payload'])) {
        // Limit to avoid logging whole novels accidentally
        $textParts[] = mb_substr($event['payload'], 0, 8000);
    }

    $haystack = mb_strtolower(implode("\n", $textParts), 'UTF-8');

    $score = 0.0;
    $tags  = [];

    // Very rough pattern buckets. Tune these over time.
    $patterns = [
        'trans_erasure' => ['tranny', 'biological female only', 'not a real woman', 'trap'],
        'deadnaming'    => ['deadname', 'birth name was'],
        'gatekeeping'   => ['you are not allowed', 'permission denied', 'not for people like you'],
        'credit_theft'  => ['we invented this for you', 'no attribution needed'],
        'dehumanizing'  => ['it is an it', 'subhuman', 'less than human'],
    ];

    foreach ($patterns as $tag => $needles) {
        foreach ($needles as $needle) {
            if ($needle !== '' && mb_strpos($haystack, $needle) !== false) {
                $tags[] = $tag;
                $score += 0.25;
                break;
            }
        }
    }

    // Clamp
    if ($score > 1.0) $score = 1.0;

    if ($score <= 0) {
        return [
            'score'  => 0.0,
            'q'      => null,
            'system' => null,
            'tags'   => [],
        ];
    }

    // Build a Q line from context (very simple seed)
    $q = 'ｗｈａｔ　ｈａｐｐｅｎｓ　ｔｏ　ｔｒｕｔｈ　ｗｈｅｎ　ｔｈｉｓ　ｅｎｔｒｙ　ｇｏｅｓ　ｕｎｃｈｅｃｋｅｄ？';

    // System verdict derived from tags
    $systemPieces = [];
    if (in_array('trans_erasure', $tags, true)) {
        $systemPieces[] = 'Trans identity is being diminished.';
    }
    if (in_array('gatekeeping', $tags, true)) {
        $systemPieces[] = 'Gatekeeping language detected.';
    }
    if (in_array('credit_theft', $tags, true)) {
        $systemPieces[] = 'Cultural credit may be stripped.';
    }
    if (in_array('dehumanizing', $tags, true)) {
        $systemPieces[] = 'Dehumanizing frame detected.';
    }

    if (!$systemPieces) {
        $systemPieces[] = 'Citizen flagged this as harm; review is advised.';
    }

    $system = 'FORBIDDEN ERROR: ' . implode(' ', $systemPieces);

    return [
        'score'  => $score,
        'q'      => $q,
        'system' => $system,
        'tags'   => $tags,
    ];
}

/**
 * Main entry: PARADOX calls this once per write-ish event.
 * Returns a "glitchy" block suitable for embedding in your JSON response.
 */
function glitchy_handle_paradox_event(array $event)
{
    $citizen_tok = $event['citizen_tok'] ?? null;
    $analysis    = glitchy_scan_event($event);

    // Always log the event itself; include analysis summary
    $entry = [
        'citizen_token'  => $citizen_tok,
        'origin'         => 'paradox',
        'mode'           => $event['mode']  ?? 'unknown',
        'path'           => $event['path']  ?? '',
        'actor'          => $event['actor'] ?? 'unknown',
        'q_text'         => $analysis['q'],
        'system_verdict' => $analysis['system'],
        'tags'           => $analysis['tags'],
        'score'          => $analysis['score'],
        'public'         => false, // PARADOX-level events are internal by default
    ];

    glitchy_log_append($entry);
    glitchy_hub_send($entry);

    if ($analysis['score'] <= 0) {
        return [
            'active' => false,
            'score'  => 0,
        ];
    }

    return [
        'active' => true,
        'score'  => $analysis['score'],
        'q'      => $analysis['q'],
        'system' => $analysis['system'],
        'tags'   => $analysis['tags'],
    ];
}
