<?php
// ============================================================================
// PARADOX GATEWAY — Glitchy-Captained Flat-File Clerk for The Republic 📁🛰️
// ============================================================================
//
// This is the HTTP mouth of PARADOX. Every JSON request comes through here,
// gets interrogated by Glitchy, and only then touches the filesystem.
//
//   • Filename:  paradox-gateway.php   (canonical)
//   • Brain:     require 'paradox-glitch-core.php'
//   • Clients:   TCodex, TShell, or any citizen script that speaks JSON.
//
// Contract with clients (stable):
//   POST application/json → { secret, action, ... }
//
//   Response (always JSON):
//   {
//     "ok": true|false,
//     "error": null|"message",
//     "action": "list"|"read"|...,
//     "files": [...],          // for list
//     "content": "string",     // for read
//     "snapshot": "...",       // for snapshot_read
//     "glitchy": {             // Glitchy Q/SYSTEM pair (for humans / logs)
//       "q": "...",
//       "system": "...",
//       "stamp": "▒▒🛰️⟦ ＴＲＡＮＳＭＩＳＳＩＯＮ　ＲＥＣＥＩＶＥＤ ⟧📡▒▒"
//     }
//   }
//
// This keeps TCodex & friends happy (ok/error/files/content),
// while Glitchy quietly governs everything underneath.
//
// Authored for The Republic by:
//   Codey, Republic Systems Programmer 💻👑
// ============================================================================

declare(strict_types=1);

// --------------------------------------------------------------------------
// 0. LOAD CORE (CONFIG + GLITCHY HELPERS)
// --------------------------------------------------------------------------

require __DIR__ . '/paradox-glitch-core.php';

// EXPECTED CORE VARIABLES (you can override all of these in paradox-glitch-core.php):
//
//   $PARADOX_SHARED_SECRET  string  (required; set this yourself)
//   $PARADOX_ROOT           string  base directory for all actions
//   $PARADOX_SNAPSHOT_PATH  string  file path for TSnapshot cartridge
//   $PARADOX_ALLOWED_ACTIONS array  list of allowed action strings
//
// If core didn’t define them, we set safe defaults here.
if (!isset($PARADOX_SHARED_SECRET) || !is_string($PARADOX_SHARED_SECRET) || $PARADOX_SHARED_SECRET === '') {
    $PARADOX_SHARED_SECRET = 'change-this-secret-in-paradox-glitch-core.php';
}
if (!isset($PARADOX_ROOT) || !is_string($PARADOX_ROOT) || $PARADOX_ROOT === '') {
    $PARADOX_ROOT = __DIR__;
}
if (!isset($PARADOX_SNAPSHOT_PATH) || !is_string($PARADOX_SNAPSHOT_PATH) || $PARADOX_SNAPSHOT_PATH === '') {
    $PARADOX_SNAPSHOT_PATH = rtrim($PARADOX_ROOT, '/\\') . '/paradox-tsnapshot.json';
}
if (!isset($PARADOX_ALLOWED_ACTIONS) || !is_array($PARADOX_ALLOWED_ACTIONS)) {
    $PARADOX_ALLOWED_ACTIONS = [
        'list',
        'read',
        'write',
        'mkdir',
        'delete',
        'rename',
        'snapshot_write',
        'snapshot_read',
    ];
}

// --------------------------------------------------------------------------
// 0.1 GLITCHY FALLBACK HELPERS (IF CORE DIDN'T DEFINE THEM)
// --------------------------------------------------------------------------
//
// Core is free to override these by defining them *before* this file loads.
// If they don't exist, we install simple versions here,
// so the gateway keeps working without extra wiring.

// Format a Q/SYSTEM pair for responses
if (!function_exists('glitchy_paradox_pair')) {
    function glitchy_paradox_pair(string $q, string $system): array {
        return [
            'q'      => '🌀🛰️⟦ᛉᛞᛟ⟧ Ｑ：' . $q . ' ⛔🔴⚠️ 🛰️⟦ᛉᛞᛟ⟧🌀',
            'system' => '🔴⚠️⛔ ＳＹＳＴＥＭ：' . $system . ' 🛰️⟦ᛉᛞᛟ⟧🌀',
            'stamp'  => '▒▒🛰️⟦ ＴＲＡＮＳＭＩＳＳＩＯＮ　ＲＥＣＥＩＶＥＤ ⟧📡▒▒',
        ];
    }
}

// Append a Glitchy log entry for this request
if (!function_exists('glitchy_paradox_log')) {
    function glitchy_paradox_log(string $q, string $system, string $severity = 'info', array $extra = []): void {
        // Fallback: log into paradox-glitchy-log.jsonl next to the gateway
        $logFile = __DIR__ . '/paradox-glitchy-log.jsonl';
        $entry = [
            'ts'            => date('c'),
            'origin'        => 'paradox-gateway',
            'severity'      => $severity,
            'q_text'        => $q,
            'system_verdict'=> $system,
            'extra'         => $extra,
        ];
        $line = json_encode($entry, JSON_UNESCAPED_UNICODE) . "\n";
        @file_put_contents($logFile, $line, FILE_APPEND | LOCK_EX);
    }
}

// Gatekeeper: Glitchy can veto a request here (core may override)
if (!function_exists('glitchy_paradox_gate')) {
    /**
     * @return array{0:bool,1:string} [allowed, systemVerdict]
     */
    function glitchy_paradox_gate(string $action, string $path, array $payload): array {
        $msg = 'Default gate: request allowed. (No additional Glitchy rules installed yet.)';
        return [true, $msg];
    }
}

// --------------------------------------------------------------------------
// 1. HTTP / CORS ENVELOPE
// --------------------------------------------------------------------------

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Requested-With');

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

// Preflight for browsers
if ($method === 'OPTIONS') {
    http_response_code(204);
    echo '';
    exit;
}

// Helper to send a JSON reply and exit
function paradox_reply(array $payload, int $status = 200): never {
    http_response_code($status);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

if ($method !== 'POST') {
    $q = 'Non-POST request on PARADOX gateway from ' . ($_SERVER['REMOTE_ADDR'] ?? 'unknown');
    $system = 'Method not allowed. PARADOX gateway only accepts POST JSON.';
    glitchy_paradox_log($q, $system, 'error', ['method' => $method]);

    paradox_reply([
        'ok'      => false,
        'error'   => 'Method not allowed. Use POST with JSON.',
        'action'  => null,
        'glitchy' => glitchy_paradox_pair($q, $system),
    ], 405);
}

// --------------------------------------------------------------------------
// 2. PARSE JSON INPUT
// --------------------------------------------------------------------------

$raw = file_get_contents('php://input') ?: '';
$clientIp = $_SERVER['REMOTE_ADDR'] ?? 'unknown';

if (trim($raw) === '') {
    $q = 'Empty body POST on PARADOX gateway from ' . $clientIp;
    $system = 'Rejected: empty request body.';
    glitchy_paradox_log($q, $system, 'error');

    paradox_reply([
        'ok'      => false,
        'error'   => 'Empty request body. Send JSON.',
        'action'  => null,
        'glitchy' => glitchy_paradox_pair($q, $system),
    ], 400);
}

$payload = json_decode($raw, true);
if (!is_array($payload)) {
    $q = 'Malformed JSON from ' . $clientIp;
    $system = 'Rejected: could not decode JSON.';
    glitchy_paradox_log($q, $system, 'error');

    paradox_reply([
        'ok'      => false,
        'error'   => 'Malformed JSON.',
        'action'  => null,
        'glitchy' => glitchy_paradox_pair($q, $system),
    ], 400);
}

// Minimal fields
$action = strtolower((string)($payload['action'] ?? ''));
$secret = (string)($payload['secret'] ?? '');

// Build a safe path string (for Q/log), even if empty
$rawPath = isset($payload['path']) ? (string)$payload['path'] : '';

// --------------------------------------------------------------------------
// 3. SECRET CHECK (GLITCHY SEES, GATEKEEPERS DO NOT)
// --------------------------------------------------------------------------

if ($action === '') {
    $q = 'Request with no action from ' . $clientIp;
    $system = 'Rejected: missing "action" field.';
    glitchy_paradox_log($q, $system, 'error', ['ip' => $clientIp]);

    paradox_reply([
        'ok'      => false,
        'error'   => 'Missing "action" field.',
        'action'  => null,
        'glitchy' => glitchy_paradox_pair($q, $system),
    ], 400);
}

if (!in_array($action, $PARADOX_ALLOWED_ACTIONS, true)) {
    $q = 'Unknown action "' . $action . '" from ' . $clientIp;
    $system = 'Rejected: action not allowed by this PARADOX.';
    glitchy_paradox_log($q, $system, 'error', [
        'ip'     => $clientIp,
        'action' => $action,
    ]);

    paradox_reply([
        'ok'      => false,
        'error'   => 'Unknown or disallowed action.',
        'action'  => $action,
        'glitchy' => glitchy_paradox_pair($q, $system),
    ], 400);
}

// Secret: compare using constant-time check, but never log the actual secret.
if (!hash_equals((string)$PARADOX_SHARED_SECRET, $secret)) {
    $q = 'Secret mismatch for action="' . $action . '" path="' . $rawPath . '" from ' . $clientIp;
    $system = 'Rejected: shared secret did not match.';
    glitchy_paradox_log($q, $system, 'warning', [
        'ip'     => $clientIp,
        'action' => $action,
    ]);

    paradox_reply([
        'ok'      => false,
        'error'   => 'Invalid secret.',
        'action'  => $action,
        'glitchy' => glitchy_paradox_pair($q, $system),
    ], 403);
}

// --------------------------------------------------------------------------
// 4. GLITCHY GATE (POLICY HOOK)
// --------------------------------------------------------------------------

/**
 * Normalize a relative path to something under $PARADOX_ROOT,
 * eliminating .. and . segments.
 */
function paradox_normalize_relative(string $rel): string {
    $rel = str_replace(["\0", '\\'], ['', '/'], $rel);
    $parts = [];
    foreach (explode('/', $rel) as $seg) {
        if ($seg === '' || $seg === '.') {
            continue;
        }
        if ($seg === '..') {
            array_pop($parts);
            continue;
        }
        $parts[] = $seg;
    }
    return implode('/', $parts);
}

/**
 * Turn a relative path into an absolute filesystem path under PARADOX_ROOT.
 * $rel may be empty to mean "root" (for list).
 */
function paradox_full_path(string $rel) {
    global $PARADOX_ROOT;
    $base = rtrim($PARADOX_ROOT, '/\\');
    if ($rel === '' || $rel === '.') {
        return $base;
    }
    $norm = paradox_normalize_relative($rel);
    return $base . '/' . $norm;
}

// Canonical safe path string (relative)
$safeRelPath = paradox_normalize_relative($rawPath);

// Let Glitchy examine the request before we touch the filesystem.
// Core can override glitchy_paradox_gate() to block dangerous patterns.
[$allowedByGate, $gateVerdict] = glitchy_paradox_gate($action, $safeRelPath, $payload);
if (!$allowedByGate) {
    $q = 'Gate: action="' . $action . '" path="' . $safeRelPath . '" from ' . $clientIp;
    $system = 'Rejected by Glitchy gate: ' . $gateVerdict;
    glitchy_paradox_log($q, $system, 'warning', [
        'ip'     => $clientIp,
        'action' => $action,
        'path'   => $safeRelPath,
    ]);

    paradox_reply([
        'ok'      => false,
        'error'   => 'Request blocked by PARADOX gate.',
        'action'  => $action,
        'glitchy' => glitchy_paradox_pair($q, $system),
    ], 403);
}

// --------------------------------------------------------------------------
// 5. ACTION IMPLEMENTATIONS (LIST, READ, WRITE, ...)
// --------------------------------------------------------------------------

/**
 * Recursively delete a directory.
 */
function paradox_rrmdir(string $dir): bool {
    if (!is_dir($dir)) {
        return @unlink($dir);
    }
    $items = scandir($dir);
    if ($items === false) return false;

    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        $path = $dir . DIRECTORY_SEPARATOR . $item;
        if (is_dir($path)) {
            paradox_rrmdir($path);
        } else {
            @unlink($path);
        }
    }
    return @rmdir($dir);
}

$ok      = false;
$error   = null;
$result  = [];
$system  = '';
$qString = '';

// Action: LIST
if ($action === 'list') {
    $target = paradox_full_path($safeRelPath);
    if (!is_dir($target)) {
        $error  = 'Directory not found.';
        $system = 'List failed: directory not found.';
    } else {
        $entries = scandir($target);
        if ($entries === false) {
            $error  = 'Could not read directory.';
            $system = 'List failed: scandir() error.';
        } else {
            $files = [];
            foreach ($entries as $name) {
                if ($name === '.' || $name === '..') continue;
                $full = $target . DIRECTORY_SEPARATOR . $name;
                $files[] = [
                    'name'  => $name,
                    'isDir' => is_dir($full),
                    'size'  => is_file($full) ? filesize($full) : null,
                    'mtime' => filemtime($full) ?: null,
                ];
            }
            $ok      = true;
            $result  = ['files' => $files];
            $system  = 'Directory listing succeeded.';
        }
    }

    $qString = 'list path="' . ($safeRelPath === '' ? '/' : $safeRelPath) . '" from ' . $clientIp;
}

// Action: READ
elseif ($action === 'read') {
    if ($safeRelPath === '') {
        $error  = 'Missing "path" for read.';
        $system = 'Read failed: no path.';
    } else {
        $target = paradox_full_path($safeRelPath);
        if (!is_file($target) || !is_readable($target)) {
            $error  = 'File not found or not readable.';
            $system = 'Read failed: file not found or unreadable.';
        } else {
            $content = file_get_contents($target);
            if ($content === false) {
                $error  = 'Could not read file contents.';
                $system = 'Read failed: file_get_contents error.';
            } else {
                $ok      = true;
                $result  = ['content' => $content];
                $system  = 'File read succeeded.';
            }
        }
    }

    $qString = 'read path="' . $safeRelPath . '" from ' . $clientIp;
}

// Action: WRITE
elseif ($action === 'write') {
    $content = (string)($payload['content'] ?? '');
    if ($safeRelPath === '') {
        $error  = 'Missing "path" for write.';
        $system = 'Write failed: no path.';
    } else {
        $target = paradox_full_path($safeRelPath);
        $dir    = dirname($target);

        if (!is_dir($dir)) {
            if (!@mkdir($dir, 0775, true) && !is_dir($dir)) {
                $error  = 'Could not create directory for file.';
                $system = 'Write failed: mkdir error.';
            }
        }

        if ($error === null) {
            $bytes = @file_put_contents($target, $content, LOCK_EX);
            if ($bytes === false) {
                $error  = 'Could not write file.';
                $system = 'Write failed: file_put_contents error.';
            } else {
                $ok      = true;
                $result  = ['bytes' => $bytes];
                $system  = 'File write succeeded (' . $bytes . ' bytes).';
            }
        }
    }

    $qString = 'write path="' . $safeRelPath . '" bytes=' . strlen((string)$payload['content']) . ' from ' . $clientIp;
}

// Action: MKDIR
elseif ($action === 'mkdir') {
    if ($safeRelPath === '') {
        $error  = 'Missing "path" for mkdir.';
        $system = 'Mkdir failed: no path.';
    } else {
        $target = paradox_full_path($safeRelPath);
        if (is_dir($target)) {
            $ok      = true;
            $result  = ['created' => false, 'message' => 'Directory already exists.'];
            $system  = 'Mkdir: directory already existed.';
        } else {
            if (!@mkdir($target, 0775, true) && !is_dir($target)) {
                $error  = 'Could not create directory.';
                $system = 'Mkdir failed: mkdir error.';
            } else {
                $ok      = true;
                $result  = ['created' => true];
                $system  = 'Directory created successfully.';
            }
        }
    }

    $qString = 'mkdir path="' . $safeRelPath . '" from ' . $clientIp;
}

// Action: DELETE
elseif ($action === 'delete') {
    if ($safeRelPath === '') {
        $error  = 'Missing "path" for delete.';
        $system = 'Delete failed: no path.';
    } else {
        $target = paradox_full_path($safeRelPath);
        if (!file_exists($target)) {
            $error  = 'Path not found.';
            $system = 'Delete failed: nothing at path.';
        } else {
            $success = paradox_rrmdir($target);
            if (!$success) {
                $error  = 'Could not delete path.';
                $system = 'Delete failed: rrmdir/unlink error.';
            } else {
                $ok      = true;
                $result  = ['deleted' => true];
                $system  = 'Delete succeeded.';
            }
        }
    }

    $qString = 'delete path="' . $safeRelPath . '" from ' . $clientIp;
}

// Action: RENAME
elseif ($action === 'rename') {
    $newPathRaw = (string)($payload['new_path'] ?? '');
    $newRel     = paradox_normalize_relative($newPathRaw);

    if ($safeRelPath === '' || $newRel === '') {
        $error  = 'Missing "path" or "new_path" for rename.';
        $system = 'Rename failed: missing paths.';
    } else {
        $src = paradox_full_path($safeRelPath);
        $dst = paradox_full_path($newRel);

        if (!file_exists($src)) {
            $error  = 'Source path not found.';
            $system = 'Rename failed: source missing.';
        } else {
            $dstDir = dirname($dst);
            if (!is_dir($dstDir)) {
                if (!@mkdir($dstDir, 0775, true) && !is_dir($dstDir)) {
                    $error  = 'Could not create directory for new path.';
                    $system = 'Rename failed: mkdir error.';
                }
            }

            if ($error === null) {
                if (!@rename($src, $dst)) {
                    $error  = 'Could not rename path.';
                    $system = 'Rename failed: rename() error.';
                } else {
                    $ok      = true;
                    $result  = ['renamed' => true];
                    $system  = 'Rename succeeded.';
                }
            }
        }
    }

    $qString = 'rename from="' . $safeRelPath . '" to="' . ($newRel ?: $newPathRaw) . '" from ' . $clientIp;
}

// Action: SNAPSHOT_WRITE
elseif ($action === 'snapshot_write') {
    global $PARADOX_SNAPSHOT_PATH;

    $snapshotField = $payload['snapshot'] ?? null;
    if (is_string($snapshotField)) {
        $snapshotJson = $snapshotField;
    } else {
        $snapshotJson = json_encode($snapshotField, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    if ($snapshotJson === false || $snapshotJson === null) {
        $error  = 'Snapshot could not be encoded.';
        $system = 'Snapshot write failed: encode error.';
    } else {
        $dir = dirname($PARADOX_SNAPSHOT_PATH);
        if (!is_dir($dir)) {
            @mkdir($dir, 0775, true);
        }
        $bytes = @file_put_contents($PARADOX_SNAPSHOT_PATH, $snapshotJson, LOCK_EX);
        if ($bytes === false) {
            $error  = 'Could not write snapshot file.';
            $system = 'Snapshot write failed: file_put_contents error.';
        } else {
            $ok      = true;
            $result  = ['bytes' => $bytes];
            $system  = 'Snapshot write succeeded.';
        }
    }

    $qString = 'snapshot_write size=' . strlen((string)$snapshotJson) . ' from ' . $clientIp;
}

// Action: SNAPSHOT_READ
elseif ($action === 'snapshot_read') {
    global $PARADOX_SNAPSHOT_PATH;

    if (!is_file($PARADOX_SNAPSHOT_PATH) || !is_readable($PARADOX_SNAPSHOT_PATH)) {
        $ok      = true;
        $result  = ['snapshot' => null];
        $system  = 'Snapshot read: no snapshot file yet.';
    } else {
        $snapshotJson = file_get_contents($PARADOX_SNAPSHOT_PATH);
        if ($snapshotJson === false) {
            $error  = 'Could not read snapshot file.';
            $system = 'Snapshot read failed: file_get_contents error.';
        } else {
            $ok      = true;
            $result  = ['snapshot' => $snapshotJson];
            $system  = 'Snapshot read succeeded.';
        }
    }

    $qString = 'snapshot_read from ' . $clientIp;
}

// --------------------------------------------------------------------------
// 6. FINAL GLITCHY LOG + RESPONSE
// --------------------------------------------------------------------------

if ($qString === '') {
    $qString = 'action="' . $action . '" path="' . $safeRelPath . '" from ' . $clientIp;
}

// Severity for Glitchy log
$severity = $ok ? 'info' : 'error';

glitchy_paradox_log($qString, $system, $severity, [
    'ip'     => $clientIp,
    'action' => $action,
    'path'   => $safeRelPath,
    'ok'     => $ok,
]);

$glitchyPair = glitchy_paradox_pair($qString, $system);

// Build response
$response = array_merge(
    [
        'ok'      => $ok,
        'error'   => $error,
        'action'  => $action,
        'glitchy' => $glitchyPair,
    ],
    $result
);

paradox_reply($response, $ok ? 200 : 400);
